<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'teacher') {
    header("Location: ../login.php");
    exit();
}

$teacher_id = $_SESSION['user_id'];
$assignment_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Initialize variables
$assignment = null;
$submissions = [];
$success = '';
$error = '';

// Fetch assignment details
if ($assignment_id > 0) {
    $assignment_query = "SELECT a.*, c.class_name, s.section_name, sub.subject_name, u.full_name as teacher_name
                         FROM assignments a
                         JOIN classes c ON a.class_id = c.id
                         JOIN sections s ON a.section_id = s.id
                         JOIN subjects sub ON a.subject_id = sub.id
                         JOIN users u ON a.teacher_id = u.id
                         WHERE a.id = $assignment_id AND a.teacher_id = $teacher_id";
    
    $assignment_result = mysqli_query($conn, $assignment_query);
    
    if ($assignment_result && mysqli_num_rows($assignment_result) > 0) {
        $assignment = mysqli_fetch_assoc($assignment_result);
        
        // Fetch submissions for this assignment
        $submissions_query = "SELECT asub.*, s.full_name as student_name, s.student_id
                              FROM assignment_submissions asub
                              JOIN students s ON asub.student_id = s.id
                              WHERE asub.assignment_id = $assignment_id
                              ORDER BY asub.submitted_at DESC";
        $submissions_result = mysqli_query($conn, $submissions_query);
        
        if ($submissions_result && mysqli_num_rows($submissions_result) > 0) {
            while ($submission = mysqli_fetch_assoc($submissions_result)) {
                $submissions[] = $submission;
            }
        }
    } else {
        $error = "Assignment not found or you don't have permission to view it.";
    }
} else {
    $error = "Invalid assignment ID.";
}

// Handle grading submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['grade_assignment'])) {
    $submission_id = intval($_POST['submission_id']);
    $grade = floatval($_POST['grade']);
    $feedback = mysqli_real_escape_string($conn, $_POST['feedback']);
    
    // Validate grade
    if ($grade < 0 || $grade > $assignment['max_marks']) {
        $error = "Grade must be between 0 and " . $assignment['max_marks'];
    } else {
        $update_query = "UPDATE assignment_submissions 
                         SET grade = $grade, feedback = '$feedback', graded_at = NOW() 
                         WHERE id = $submission_id";
        
        if (mysqli_query($conn, $update_query)) {
            $success = "Assignment graded successfully!";
            
            // Refresh submissions data
            $submissions_result = mysqli_query($conn, $submissions_query);
            $submissions = [];
            if ($submissions_result && mysqli_num_rows($submissions_result) > 0) {
                while ($submission = mysqli_fetch_assoc($submissions_result)) {
                    $submissions[] = $submission;
                }
            }
        } else {
            $error = "Failed to grade assignment. Please try again.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Assignment - Teacher Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50">
<div class="container mx-auto px-4 py-8">
    <?php if (!empty($error)): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
            <?php echo $error; ?>
        </div>
    <?php endif; ?>
    
    <?php if (!empty($success)): ?>
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
            <?php echo $success; ?>
        </div>
    <?php endif; ?>

    <?php if ($assignment): ?>
    <div class="bg-white p-6  ">
        <!-- Header -->
        <div class="flex justify-between items-center mb-6">
            <div>
                <h1 class="text-2xl font-bold text-gray-800"><?php echo htmlspecialchars($assignment['title']); ?></h1>
                <p class="text-gray-600"><?php echo htmlspecialchars($assignment['subject_name']); ?> • <?php echo htmlspecialchars($assignment['class_name']); ?> - <?php echo htmlspecialchars($assignment['section_name']); ?></p>
            </div>
            <a href="assigment.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 ">
                <i class="fas fa-arrow-left mr-2"></i> Back to Assignments
            </a>
        </div>

        <!-- Assignment Details -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
            <div class="bg-gray-50 p-4 ">
                <h2 class="text-lg font-semibold text-gray-800 mb-3">Assignment Details</h2>
                <div class="space-y-3">
                    <div>
                        <p class="text-sm text-gray-600">Teacher</p>
                        <p class="font-medium"><?php echo htmlspecialchars($assignment['teacher_name']); ?></p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Due Date</p>
                        <p class="font-medium <?php echo (strtotime($assignment['due_date']) < time()) ? 'text-red-600' : 'text-gray-800'; ?>">
                            <?php echo date('M j, Y g:i A', strtotime($assignment['due_date'])); ?>
                            <?php if (strtotime($assignment['due_date']) < time()): ?>
                                <span class="text-red-500 text-sm">(Overdue)</span>
                            <?php endif; ?>
                        </p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Maximum Marks</p>
                        <p class="font-medium"><?php echo $assignment['max_marks'] > 0 ? $assignment['max_marks'] : 'Not specified'; ?></p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Created On</p>
                        <p class="font-medium"><?php echo date('M j, Y g:i A', strtotime($assignment['created_at'])); ?></p>
                    </div>
                </div>
            </div>

            <div class="bg-gray-50 p-4 ">
                <h2 class="text-lg font-semibold text-gray-800 mb-3">Submission Status</h2>
                <div class="space-y-3">
                    <?php
                    // Get total students in the class
                    $total_students_query = "SELECT COUNT(*) as total FROM students 
                                            WHERE class = {$assignment['class_id']} 
                                            AND section = {$assignment['section_id']} 
                                            AND status = 'active'";
                    $total_students_result = mysqli_query($conn, $total_students_query);
                    $total_students = mysqli_fetch_assoc($total_students_result)['total'];
                    
                    $submitted_count = count($submissions);
                    $graded_count = 0;
                    
                    foreach ($submissions as $submission) {
                        if (!is_null($submission['grade'])) {
                            $graded_count++;
                        }
                    }
                    ?>
                    <div>
                        <p class="text-sm text-gray-600">Total Students</p>
                        <p class="font-medium"><?php echo $total_students; ?></p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Submissions Received</p>
                        <p class="font-medium"><?php echo $submitted_count; ?> (<?php echo round(($submitted_count / $total_students) * 100, 1); ?>%)</p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Assignments Graded</p>
                        <p class="font-medium"><?php echo $graded_count; ?> (<?php echo $submitted_count > 0 ? round(($graded_count / $submitted_count) * 100, 1) : 0; ?>%)</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Assignment Description & File -->
        <div class="mb-8">
            <?php if (!empty($assignment['description'])): ?>
                <div class="mb-4">
                    <h2 class="text-lg font-semibold text-gray-800 mb-2">Description</h2>
                    <div class="bg-gray-50 p-4 ">
                        <p class="text-gray-800"><?php echo nl2br(htmlspecialchars($assignment['description'])); ?></p>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($assignment['file_path'])): ?>
                <div>
                    <h2 class="text-lg font-semibold text-gray-800 mb-2">Assignment File</h2>
                    <a href="../uploads/assignments/<?php echo htmlspecialchars($assignment['file_path']); ?>" 
                       class="inline-flex items-center bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 " download>
                        <i class="fas fa-download mr-2"></i> Download Assignment File
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <!-- Student Submissions -->
        <div>
            <h2 class="text-xl font-semibold text-gray-800 mb-4">Student Submissions</h2>
            
            <?php if (count($submissions) > 0): ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full bg-white  overflow-hidden">
                        <thead class="bg-gray-200 text-gray-700">
                            <tr>
                                <th class="py-3 px-4 text-left">Student</th>
                                <th class="py-3 px-4 text-left">Submitted On</th>
                                <th class="py-3 px-4 text-left">Answer</th>
                                <th class="py-3 px-4 text-left">File</th>
                                <th class="py-3 px-4 text-left">Grade</th>
                                <th class="py-3 px-4 text-left">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="text-gray-600">
                            <?php foreach ($submissions as $submission): ?>
                                <tr class="border-b border-gray-200 hover:bg-gray-50">
                                    <td class="py-3 px-4">
                                        <div>
                                            <p class="font-medium"><?php echo htmlspecialchars($submission['student_name']); ?></p>
                                            <p class="text-sm text-gray-500">ID: <?php echo htmlspecialchars($submission['student_id']); ?></p>
                                        </div>
                                    </td>
                                    <td class="py-3 px-4">
                                        <?php echo date('M j, Y g:i A', strtotime($submission['submitted_at'])); ?>
                                    </td>
                                    <td class="py-3 px-4">
                                        <?php if (!empty($submission['submission_text'])): ?>
                                            <button onclick="toggleAnswer(<?php echo $submission['id']; ?>)" 
                                                    class="text-blue-600 hover:text-blue-800 text-sm">
                                                View Answer
                                            </button>
                                            <div id="answer-<?php echo $submission['id']; ?>" class="hidden mt-2 p-3 bg-gray-100 ">
                                                <?php echo nl2br(htmlspecialchars($submission['submission_text'])); ?>
                                            </div>
                                        <?php else: ?>
                                            <span class="text-gray-400">No text answer</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="py-3 px-4">
                                        <?php if (!empty($submission['submission_file'])): ?>
                                            <a href="../uploads/assignment_answers/<?php echo htmlspecialchars($submission['submission_file']); ?>" 
                                               class="text-blue-600 hover:text-blue-800" download>
                                                <i class="fas fa-download mr-1"></i> Download
                                            </a>
                                        <?php else: ?>
                                            <span class="text-gray-400">No file</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="py-3 px-4">
                                        <?php if (!is_null($submission['grade'])): ?>
                                            <span class="font-medium"><?php echo $submission['grade']; ?>/<?php echo $assignment['max_marks']; ?></span>
                                            <?php if (!empty($submission['feedback'])): ?>
                                                <button onclick="toggleFeedback(<?php echo $submission['id']; ?>)" 
                                                        class="ml-2 text-blue-600 hover:text-blue-800 text-sm">
                                                    View Feedback
                                                </button>
                                                <div id="feedback-<?php echo $submission['id']; ?>" class="hidden mt-2 p-3 bg-gray-100 ">
                                                    <?php echo nl2br(htmlspecialchars($submission['feedback'])); ?>
                                                </div>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="text-gray-400">Not graded</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="py-3 px-4">
                                        <button onclick="openGradeModal(<?php echo $submission['id']; ?>, '<?php echo htmlspecialchars($submission['student_name']); ?>', <?php echo $submission['grade'] ?? 'null'; ?>, `<?php echo isset($submission['feedback']) ? htmlspecialchars($submission['feedback']) : ''; ?>`)" 
                                                class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded text-sm">
                                            <i class="fas fa-edit mr-1"></i> Grade
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="bg-gray-100 p-6  text-center">
                    <i class="fas fa-inbox text-4xl text-gray-300 mb-3"></i>
                    <h3 class="text-lg font-medium text-gray-600">No submissions yet</h3>
                    <p class="text-gray-500">Students haven't submitted their work for this assignment.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Grading Modal -->
    <div id="gradeModal" class="fixed inset-0 bg-gray-800 bg-opacity-75 flex items-center justify-center z-50 hidden">
        <div class="bg-white  shadow-xl w-full max-w-md mx-4">
            <div class="p-6">
                <h3 class="text-xl font-semibold text-gray-800 mb-2">Grade Assignment</h3>
                <p class="text-gray-600 mb-4">Student: <span id="modalStudentName"></span></p>
                
                <form method="POST" action="">
                    <input type="hidden" name="submission_id" id="modalSubmissionId">
                    
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Grade (out of <?php echo $assignment['max_marks']; ?>)</label>
                        <input type="number" name="grade" id="modalGrade" step="0.01" min="0" max="<?php echo $assignment['max_marks']; ?>" 
                               class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                    </div>
                    
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Feedback (optional)</label>
                        <textarea name="feedback" id="modalFeedback" rows="4" 
                                  class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
                    </div>
                    
                    <div class="flex justify-end space-x-2">
                        <button type="button" onclick="closeGradeModal()" 
                                class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 ">
                            Cancel
                        </button>
                        <button type="submit" name="grade_assignment" 
                                class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 ">
                            Save Grade
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
    function toggleAnswer(submissionId) {
        const answerDiv = document.getElementById('answer-' + submissionId);
        answerDiv.classList.toggle('hidden');
    }
    
    function toggleFeedback(submissionId) {
        const feedbackDiv = document.getElementById('feedback-' + submissionId);
        feedbackDiv.classList.toggle('hidden');
    }
    
    function openGradeModal(submissionId, studentName, currentGrade, currentFeedback) {
        document.getElementById('modalSubmissionId').value = submissionId;
        document.getElementById('modalStudentName').textContent = studentName;
        document.getElementById('modalGrade').value = currentGrade !== null ? currentGrade : '';
        document.getElementById('modalFeedback').value = currentFeedback !== null ? currentFeedback : '';
        document.getElementById('gradeModal').classList.remove('hidden');
    }
    
    function closeGradeModal() {
        document.getElementById('gradeModal').classList.add('hidden');
    }
    
    // Close modal when clicking outside
    document.getElementById('gradeModal').addEventListener('click', function(e) {
        if (e.target.id === 'gradeModal') {
            closeGradeModal();
        }
    });
    </script>

    <?php else: ?>
        <div class="bg-white p-6   text-center">
            <i class="fas fa-exclamation-triangle text-4xl text-yellow-500 mb-4"></i>
            <h2 class="text-xl font-semibold text-gray-800 mb-2">Assignment Not Found</h2>
            <p class="text-gray-600">The assignment you're looking for doesn't exist or you don't have permission to view it.</p>
            <a href="assignments.php" class="inline-block mt-4 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 ">
                <i class="fas fa-arrow-left mr-2"></i> Back to Assignments
            </a>
        </div>
    <?php endif; ?>
</div>
</body>
</html>

<?php
include 'include/footer.php';
ob_end_flush();
?>