<?php
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
    header('Location: ../portal_login.php');
    exit();
}

$teacher_id = $_SESSION['user_id'];
$success = '';
$error = '';

// Get teacher's assigned classes and sections
$teacher_classes = [];
$teacher_sections = [];

// Query to get classes where this teacher is the class teacher
$class_query = "SELECT id, class_name FROM classes WHERE class_teacher_id = ? AND status = 'active'";
$class_stmt = $conn->prepare($class_query);
$class_stmt->bind_param("i", $teacher_id);
$class_stmt->execute();
$class_result = $class_stmt->get_result();

while ($class = $class_result->fetch_assoc()) {
    $teacher_classes[] = $class;
}
$class_stmt->close();

// Query to get sections where this teacher is the section teacher
$section_query = "SELECT s.id, s.section_name, c.class_name 
                 FROM sections s 
                 JOIN classes c ON s.class_id = c.id 
                 WHERE s.section_teacher_id = ? AND s.status = 'active'";
$section_stmt = $conn->prepare($section_query);
$section_stmt->bind_param("i", $teacher_id);
$section_stmt->execute();
$section_result = $section_stmt->get_result();

while ($section = $section_result->fetch_assoc()) {
    $teacher_sections[] = $section;
}
$section_stmt->close();

// Build the base query to get students based on teacher's classes/sections
$base_query = "SELECT s.*, c.class_name, sec.section_name 
               FROM students s 
               LEFT JOIN classes c ON s.class = c.id 
               LEFT JOIN sections sec ON s.section = sec.id 
               WHERE s.status = 'active'";

$conditions = [];
$params = [];
$types = "";

// If teacher has assigned classes, add condition
if (!empty($teacher_classes)) {
    $class_ids = array_column($teacher_classes, 'id');
    $placeholders = implode(',', array_fill(0, count($class_ids), '?'));
    $conditions[] = "s.class IN ($placeholders)";
    $params = array_merge($params, $class_ids);
    $types .= str_repeat('i', count($class_ids));
}

// If teacher has assigned sections, add condition
if (!empty($teacher_sections)) {
    $section_ids = array_column($teacher_sections, 'id');
    $placeholders = implode(',', array_fill(0, count($section_ids), '?'));
    $conditions[] = "s.section IN ($placeholders)";
    $params = array_merge($params, $section_ids);
    $types .= str_repeat('i', count($section_ids));
}

// If no classes or sections assigned, show empty result
if (empty($conditions)) {
    $query = $base_query . " AND 1=0"; // Force no results
    $count_query = $base_query . " AND 1=0";
} else {
    $query = $base_query . " AND (" . implode(" OR ", $conditions) . ")";
    $count_query = $base_query . " AND (" . implode(" OR ", $conditions) . ")";
}

// Pagination setup
$per_page = 10; // Number of records per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) $page = 1;

// Get total number of records
$count_stmt = $conn->prepare($count_query);
if (!empty($params)) {
    $count_stmt->bind_param($types, ...$params);
}
$count_stmt->execute();
$count_result = $count_stmt->get_result();
$total_records = $count_result->num_rows;
$total_pages = ceil($total_records / $per_page);

// Ensure page is within valid range
if ($page > $total_pages && $total_pages > 0) $page = $total_pages;

// Calculate offset
$offset = max(0, ($page - 1) * $per_page);

// Add pagination to the main query
$query .= " ORDER BY c.class_name, sec.section_name, s.roll_number LIMIT ?, ?";
$params[] = $offset;
$params[] = $per_page;
$types .= "ii";

// Prepare and execute the query
$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();
?>

<div class="bg-white p-6  ">
    <div class="flex justify-between items-center mb-6">
        <h2 class="text-2xl font-bold text-gray-800">My Students</h2>
        <div class="flex items-center space-x-2">
            <span class="bg-blue-100 text-blue-800 text-sm px-3 py-1 rounded-full">
                <i class="fas fa-chalkboard-teacher mr-1"></i> Teacher View
            </span>
        </div>
    </div>

    <!-- Teacher's Assigned Classes/Sections Summary -->
    <div class="mb-6 bg-blue-50 p-4 rounded-lg">
        <h3 class="text-lg font-semibold text-blue-800 mb-3">My Teaching Assignments</h3>
        
        <?php if (!empty($teacher_classes) || !empty($teacher_sections)): ?>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <?php if (!empty($teacher_classes)): ?>
                    <div>
                        <h4 class="font-medium text-blue-700 mb-2">Class Teacher For:</h4>
                        <ul class="list-disc list-inside text-blue-600">
                            <?php foreach ($teacher_classes as $class): ?>
                                <li><?php echo htmlspecialchars($class['class_name']); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($teacher_sections)): ?>
                    <div>
                        <h4 class="font-medium text-blue-700 mb-2">Section Teacher For:</h4>
                        <ul class="list-disc list-inside text-blue-600">
                            <?php foreach ($teacher_sections as $section): ?>
                                <li><?php echo htmlspecialchars($section['class_name'] . ' - ' . $section['section_name']); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <p class="text-blue-600">You are not currently assigned as a class teacher or section teacher.</p>
        <?php endif; ?>
    </div>

    <!-- Students List -->
    <div class="overflow-x-auto">
        <?php if ($result->num_rows > 0): ?>
            <table class="min-w-full bg-white">
                <thead class="bg-gray-200 text-gray-700">
                    <tr>
                        <th class="py-3 px-4 text-left">Student ID</th>
                        <th class="py-3 px-4 text-left">Full Name</th>
                        <th class="py-3 px-4 text-left">Class</th>
                        <th class="py-3 px-4 text-left">Section</th>
                        <th class="py-3 px-4 text-left">Roll No</th>
                        <th class="py-3 px-4 text-left">Email</th>
                        <th class="py-3 px-4 text-left">Phone</th>
                        <th class="py-3 px-4 text-left">Actions</th>
                    </tr>
                </thead>
                <tbody class="text-gray-600">
                    <?php while ($student = $result->fetch_assoc()): ?>
                        <tr class="border-b border-gray-200 hover:bg-gray-50">
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['student_id']); ?></td>
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['full_name']); ?></td>
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['class_name']); ?></td>
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['section_name']); ?></td>
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['roll_number']); ?></td>
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['email']); ?></td>
                            <td class="py-3 px-4"><?php echo htmlspecialchars($student['phone']); ?></td>
                            <td class="py-3 px-4">
                                <div class="flex space-x-2">
                                    <a href="view_student.php?id=<?php echo $student['id']; ?>" 
                                       class="bg-blue-500 hover:bg-blue-600 text-white px-3 py-1 rounded" title="View">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </div>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
            
            <div class="mt-4 text-sm text-gray-600">
                Showing <?php echo min($per_page, $total_records - $offset); ?> of <?php echo $total_records; ?> student(s) under your supervision
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <div class="flex justify-center mt-6">
                <nav class="inline-flex rounded-md shadow">
                    <!-- Previous Page Link -->
                    <?php if ($page > 1): ?>
                        <a href="?page=<?php echo $page - 1; ?>" class="py-2 px-4 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 rounded-l-md">
                            <i class="fas fa-chevron-left"></i>
                        </a>
                    <?php else: ?>
                        <span class="py-2 px-4 border border-gray-300 bg-gray-100 text-sm font-medium text-gray-400 rounded-l-md">
                            <i class="fas fa-chevron-left"></i>
                        </span>
                    <?php endif; ?>

                    <!-- Page Numbers -->
                    <?php 
                    // Calculate start and end of pagination links
                    $start = max(1, $page - 2);
                    $end = min($total_pages, $start + 4);
                    $start = max(1, $end - 4);
                    
                    for ($i = $start; $i <= $end; $i++): 
                    ?>
                        <?php if ($i == $page): ?>
                            <span class="py-2 px-4 border border-gray-300 bg-blue-600 text-white text-sm font-medium">
                                <?php echo $i; ?>
                            </span>
                        <?php else: ?>
                            <a href="?page=<?php echo $i; ?>" class="py-2 px-4 border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50">
                                <?php echo $i; ?>
                            </a>
                        <?php endif; ?>
                    <?php endfor; ?>

                    <!-- Next Page Link -->
                    <?php if ($page < $total_pages): ?>
                        <a href="?page=<?php echo $page + 1; ?>" class="py-2 px-4 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 rounded-r-md">
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    <?php else: ?>
                        <span class="py-2 px-4 border border-gray-300 bg-gray-100 text-sm font-medium text-gray-400 rounded-r-md">
                            <i class="fas fa-chevron-right"></i>
                        </span>
                    <?php endif; ?>
                </nav>
            </div>
            <?php endif; ?>
            
        <?php else: ?>
            <div class="text-center py-8">
                <div class="text-gray-400 text-5xl mb-4">
                    <i class="fas fa-user-graduate"></i>
                </div>
                <h3 class="text-lg font-medium text-gray-600 mb-2">No Students Found</h3>
                <p class="text-gray-500">
                    <?php if (empty($teacher_classes) && empty($teacher_sections)): ?>
                        You are not currently assigned to any classes or sections.
                    <?php else: ?>
                        No students are currently enrolled in your assigned classes/sections.
                    <?php endif; ?>
                </p>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// JavaScript for any interactive elements
document.addEventListener('DOMContentLoaded', function() {
    // Add any JavaScript functionality needed
    console.log('Teacher student page loaded');
});
</script>

<?php
include 'include/footer.php';

// Close database connections
if (isset($count_stmt)) $count_stmt->close();
if (isset($stmt)) $stmt->close();
$conn->close();
?>