<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
    header('Location: login.php');
    exit();
}

$teacher_id = $_SESSION['user_id'];

// Get filter parameters
$status_filter = isset($_GET['status']) ? mysqli_real_escape_string($conn, $_GET['status']) : 'all';
$year_filter = isset($_GET['year']) ? intval($_GET['year']) : date('Y');
$month_filter = isset($_GET['month']) ? intval($_GET['month']) : 0;

// Build query with filters
$where_conditions = ["la.teacher_id = $teacher_id"];
if ($status_filter !== 'all') {
    $where_conditions[] = "la.status = '$status_filter'";
}
if ($year_filter) {
    $where_conditions[] = "YEAR(la.start_date) = $year_filter";
}
if ($month_filter > 0) {
    $where_conditions[] = "MONTH(la.start_date) = $month_filter";
}

$where_clause = implode(' AND ', $where_conditions);

// Get leave history
$leave_query = "SELECT la.*, lt.type_name, lt.type_code,
                       COALESCE(rev.full_name, 'Not reviewed') as reviewed_by_name
                FROM leave_applications la
                JOIN leave_types lt ON la.leave_type_id = lt.id
                LEFT JOIN users rev ON la.reviewed_by = rev.id
                WHERE $where_clause
                ORDER BY la.applied_at DESC";
$leave_result = mysqli_query($conn, $leave_query);
$leave_history = [];
while ($leave = mysqli_fetch_assoc($leave_result)) {
    $leave_history[] = $leave;
}

// Get available years for filter
$years_query = "SELECT DISTINCT YEAR(applied_at) as year 
                FROM leave_applications 
                WHERE teacher_id = $teacher_id 
                ORDER BY year DESC";
$years_result = mysqli_query($conn, $years_query);
$years = [];
while ($year = mysqli_fetch_assoc($years_result)) {
    $years[] = $year['year'];
}
if (!in_array(date('Y'), $years)) {
    $years[] = date('Y');
}

// Get statistics
$stats_query = "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected,
                SUM(working_days) as total_days
                FROM leave_applications
                WHERE teacher_id = $teacher_id";
$stats_result = mysqli_query($conn, $stats_query);
$stats = mysqli_fetch_assoc($stats_result);

// Month names for filter
$month_names = [
    1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
    5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
    9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Leave History</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .leave-status {
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-size: 0.75rem;
            font-weight: 500;
        }
        .status-pending {
            background-color: #fef3c7;
            color: #92400e;
        }
        .status-approved {
            background-color: #d1fae5;
            color: #065f46;
        }
        .status-rejected {
            background-color: #fee2e2;
            color: #b91c1c;
        }
        .card {
            border-left: 4px solid;
        }
        .card-total { border-left-color: #3b82f6; }
        .card-pending { border-left-color: #f59e0b; }
        .card-approved { border-left-color: #10b981; }
        .card-rejected { border-left-color: #ef4444; }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-4 py-8">
        <div class="bg-white p-6  ">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-2xl font-bold text-gray-800">Leave History</h2>
                <a href="leave.php" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 ">
                    <i class="fas fa-plus mr-2"></i> Apply for Leave
                </a>
            </div>

            <!-- Statistics -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white p-4   card card-total">
                    <div class="text-2xl font-bold text-gray-800"><?php echo $stats['total']; ?></div>
                    <div class="text-sm text-gray-600">Total Applications</div>
                </div>
                <div class="bg-white p-4   card card-pending">
                    <div class="text-2xl font-bold text-gray-800"><?php echo $stats['pending']; ?></div>
                    <div class="text-sm text-gray-600">Pending</div>
                </div>
                <div class="bg-white p-4   card card-approved">
                    <div class="text-2xl font-bold text-gray-800"><?php echo $stats['approved']; ?></div>
                    <div class="text-sm text-gray-600">Approved</div>
                </div>
                <div class="bg-white p-4   card card-rejected">
                    <div class="text-2xl font-bold text-gray-800"><?php echo $stats['rejected']; ?></div>
                    <div class="text-sm text-gray-600">Rejected</div>
                </div>
            </div>

            <!-- Filters -->
            <div class="bg-gray-50 p-4  mb-6">
                <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                        <select name="status" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>Approved</option>
                            <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Year</label>
                        <select name="year" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <?php foreach ($years as $year): ?>
                                <option value="<?php echo $year; ?>" <?php echo $year_filter == $year ? 'selected' : ''; ?>>
                                    <?php echo $year; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Month</label>
                        <select name="month" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="0" <?php echo $month_filter == 0 ? 'selected' : ''; ?>>All Months</option>
                            <?php foreach ($month_names as $num => $name): ?>
                                <option value="<?php echo $num; ?>" <?php echo $month_filter == $num ? 'selected' : ''; ?>>
                                    <?php echo $name; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="flex items-end space-x-2">
                        <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2  h-10">
                            <i class="fas fa-filter mr-2"></i> Apply Filters
                        </button>
                        <a href="leave_history.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2  h-10">
                            <i class="fas fa-times mr-2"></i> Clear
                        </a>
                    </div>
                </form>
            </div>

            <!-- Leave History Table -->
            <div class="bg-white   overflow-hidden">
                <?php if (!empty($leave_history)): ?>
                    <div class="overflow-x-auto">
                        <table class="min-w-full">
                            <thead class="bg-gray-200 text-gray-700">
                                <tr>
                                    <th class="py-3 px-4 text-left">Leave Type</th>
                                    <th class="py-3 px-4 text-left">Dates</th>
                                    <th class="py-3 px-4 text-left">Days</th>
                                    <th class="py-3 px-4 text-left">Reason</th>
                                    <th class="py-3 px-4 text-left">Status</th>
                                    <th class="py-3 px-4 text-left">Applied On</th>
                                    <th class="py-3 px-4 text-left">Reviewed By</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-600">
                                <?php foreach ($leave_history as $leave): ?>
                                    <tr class="border-b border-gray-200 hover:bg-gray-50">
                                        <td class="py-3 px-4"><?php echo $leave['type_name']; ?></td>
                                        <td class="py-3 px-4">
                                            <?php echo date('M j, Y', strtotime($leave['start_date'])); ?> - 
                                            <?php echo date('M j, Y', strtotime($leave['end_date'])); ?>
                                        </td>
                                        <td class="py-3 px-4"><?php echo $leave['working_days']; ?></td>
                                        <td class="py-3 px-4">
                                            <div class="max-w-xs truncate" title="<?php echo htmlspecialchars($leave['reason']); ?>">
                                                <?php echo htmlspecialchars($leave['reason']); ?>
                                            </div>
                                        </td>
                                        <td class="py-3 px-4">
                                            <span class="leave-status status-<?php echo $leave['status']; ?>">
                                                <?php echo ucfirst($leave['status']); ?>
                                            </span>
                                        </td>
                                        <td class="py-3 px-4"><?php echo date('M j, Y', strtotime($leave['applied_at'])); ?></td>
                                        <td class="py-3 px-4">
                                            <?php echo $leave['reviewed_by_name']; ?>
                                            <?php if ($leave['admin_notes']): ?>
                                                <br><span class="text-xs text-gray-500">Notes: <?php echo htmlspecialchars($leave['admin_notes']); ?></span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="py-12 text-center">
                        <i class="fas fa-inbox text-4xl text-gray-300 mb-3"></i>
                        <p class="text-gray-500 text-lg">No leave applications found.</p>
                        <p class="text-gray-400 text-sm mt-1">Try adjusting your filters or apply for a new leave.</p>
                        <a href="leave.php" class="inline-block mt-4 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 ">
                            Apply for Leave
                        </a>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Export Button -->
            <?php if (!empty($leave_history)): ?>
                <div class="mt-6 flex justify-between items-center">
                    <div class="text-sm text-gray-500">
                        Showing <?php echo count($leave_history); ?> application(s)
                    </div>
                    <a href="export_leave.php?status=<?php echo $status_filter; ?>&year=<?php echo $year_filter; ?>&month=<?php echo $month_filter; ?>" 
                       class="bg-green-600 hover:bg-green-700 text-white px-4 py-2  inline-flex items-center">
                        <i class="fas fa-download mr-2"></i> Export to Excel
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
    // Function to show detailed reason in a modal
    function showReason(reason) {
        alert('Leave Reason:\n\n' + reason);
    }
    </script>
</body>
</html>

<?php
include 'include/footer.php';
ob_end_flush();
?>