<?php
date_default_timezone_set('Africa/Lagos');
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
    header('Location: ../portal_login.php');
    exit();
}

$teacher_id = $_SESSION['user_id'];
$success = '';
$error = '';

// Get teacher details
$teacher_query = "SELECT u.full_name, u.email, u.teacher_id, u.photo 
                 FROM users u 
                 WHERE u.id = $teacher_id AND u.role = 'teacher'";
$teacher_result = mysqli_query($conn, $teacher_query);
$teacher = mysqli_fetch_assoc($teacher_result);

// Get teacher's assigned classes and sections
$teacher_classes = [];
$teacher_sections = [];

// Query to get classes where this teacher is the class teacher
$class_query = "SELECT id, class_name FROM classes WHERE class_teacher_id = ? AND status = 'active'";
$class_stmt = $conn->prepare($class_query);
$class_stmt->bind_param("i", $teacher_id);
$class_stmt->execute();
$class_result = $class_stmt->get_result();

while ($class = $class_result->fetch_assoc()) {
    $teacher_classes[] = $class;
}
$class_stmt->close();

// Query to get sections where this teacher is the section teacher
$section_query = "SELECT s.id, s.section_name, c.class_name, c.id as class_id
                 FROM sections s 
                 JOIN classes c ON s.class_id = c.id 
                 WHERE s.section_teacher_id = ? AND s.status = 'active'";
$section_stmt = $conn->prepare($section_query);
$section_stmt->bind_param("i", $teacher_id);
$section_stmt->execute();
$section_result = $section_stmt->get_result();

while ($section = $section_result->fetch_assoc()) {
    $teacher_sections[] = $section;
}
$section_stmt->close();

// Get total students under teacher's supervision
$total_students = 0;
if (!empty($teacher_classes) || !empty($teacher_sections)) {
    $conditions = [];
    $params = [];
    $types = "";
    
    if (!empty($teacher_classes)) {
        $class_ids = array_column($teacher_classes, 'id');
        $placeholders = implode(',', array_fill(0, count($class_ids), '?'));
        $conditions[] = "class IN ($placeholders)";
        $params = array_merge($params, $class_ids);
        $types .= str_repeat('i', count($class_ids));
    }
    
    if (!empty($teacher_sections)) {
        $section_ids = array_column($teacher_sections, 'id');
        $placeholders = implode(',', array_fill(0, count($section_ids), '?'));
        $conditions[] = "section IN ($placeholders)";
        $params = array_merge($params, $section_ids);
        $types .= str_repeat('i', count($section_ids));
    }
    
    $student_count_query = "SELECT COUNT(*) as total FROM students WHERE status = 'active' AND (" . implode(" OR ", $conditions) . ")";
    $count_stmt = $conn->prepare($student_count_query);
    if (!empty($params)) {
        $count_stmt->bind_param($types, ...$params);
    }
    $count_stmt->execute();
    $count_result = $count_stmt->get_result();
    $total_students = $count_result->fetch_assoc()['total'];
    $count_stmt->close();
}

// Get today's attendance summary
$today = date('Y-m-d');
$attendance_summary = [
    'present' => 0,
    'absent' => 0,
    'late' => 0,
    'half_day' => 0,
    'total' => 0
];

if (!empty($teacher_classes) || !empty($teacher_sections)) {
    $conditions = [];
    $params = [];
    $types = "";
    
    if (!empty($teacher_classes)) {
        $class_ids = array_column($teacher_classes, 'id');
        $placeholders = implode(',', array_fill(0, count($class_ids), '?'));
        $conditions[] = "class_id IN ($placeholders)";
        $params = array_merge($params, $class_ids);
        $types .= str_repeat('i', count($class_ids));
    }
    
    if (!empty($teacher_sections)) {
        $section_ids = array_column($teacher_sections, 'id');
        $placeholders = implode(',', array_fill(0, count($section_ids), '?'));
        $conditions[] = "section_id IN ($placeholders)";
        $params = array_merge($params, $section_ids);
        $types .= str_repeat('i', count($section_ids));
    }
    
    $attendance_query = "SELECT status, COUNT(*) as count 
                        FROM attendance 
                        WHERE attendance_date = '$today' 
                        AND (" . implode(" OR ", $conditions) . ")
                        GROUP BY status";
    
    $attendance_stmt = $conn->prepare($attendance_query);
    if (!empty($params)) {
        $attendance_stmt->bind_param($types, ...$params);
    }
    $attendance_stmt->execute();
    $attendance_result = $attendance_stmt->get_result();
    
    while ($row = $attendance_result->fetch_assoc()) {
        $attendance_summary[$row['status']] = $row['count'];
        $attendance_summary['total'] += $row['count'];
    }
    $attendance_stmt->close();
}

// Get upcoming events
$upcoming_events_query = "SELECT * FROM events 
                         WHERE (audience_type = 'teachers' OR audience_type = 'all')
                         AND event_date >= CURDATE()
                        
                         ORDER BY event_date ASC 
                         LIMIT 5";
$upcoming_events_result = mysqli_query($conn, $upcoming_events_query);
$upcoming_events = [];
while ($event = mysqli_fetch_assoc($upcoming_events_result)) {
    $upcoming_events[] = $event;
}

// Get quiz performance data for charts
$quiz_performance_data = [];
if (!empty($teacher_classes) || !empty($teacher_sections)) {
    $conditions = [];
    $params = [];
    $types = "";
    
    if (!empty($teacher_classes)) {
        $class_ids = array_column($teacher_classes, 'id');
        $placeholders = implode(',', array_fill(0, count($class_ids), '?'));
        $conditions[] = "q.class_id IN ($placeholders)";
        $params = array_merge($params, $class_ids);
        $types .= str_repeat('i', count($class_ids));
    }
    
    if (!empty($teacher_sections)) {
        $section_ids = array_column($teacher_sections, 'id');
        $placeholders = implode(',', array_fill(0, count($section_ids), '?'));
        $conditions[] = "q.section_id IN ($placeholders)";
        $params = array_merge($params, $section_ids);
        $types .= str_repeat('i', count($section_ids));
    }
    
    // Get quiz performance data with average scores
  $performance_query = "SELECT q.quiz_title, 
                     COUNT(qa.id) as attempts
                     FROM quiz_attempts qa
                     JOIN quizzes q ON qa.quiz_id = q.id
                     WHERE (" . implode(" OR ", $conditions) . ")
                     AND qa.status = 'completed'
                     GROUP BY qa.quiz_id
                     ORDER BY qa.created_at DESC
                     LIMIT 5";

    
    $performance_stmt = $conn->prepare($performance_query);
    if (!empty($params)) {
        $performance_stmt->bind_param($types, ...$params);
    }
    $performance_stmt->execute();
    $performance_result = $performance_stmt->get_result();
    
    while ($row = $performance_result->fetch_assoc()) {
        $quiz_performance_data[] = $row;
    }
    $performance_stmt->close();
}

// Get attendance data for the chart (last 7 days)
$attendance_chart_data = [];
$attendance_dates = [];
$present_counts = [];
$absent_counts = [];

for ($i = 6; $i >= 0; $i--) {
    $date = date('Y-m-d', strtotime("-$i days"));
    $attendance_dates[] = date('M j', strtotime($date));
    
    // Get attendance data for this date
    $attendance_day_query = "SELECT 
                            SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_count,
                            SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_count,
                            SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late_count,
                            SUM(CASE WHEN status = 'half_day' THEN 1 ELSE 0 END) as half_day_count
                            FROM attendance 
                            WHERE attendance_date = '$date'";
    
    if (!empty($teacher_classes) || !empty($teacher_sections)) {
        $conditions = [];
        $params = [];
        $types = "";
        
        if (!empty($teacher_classes)) {
            $class_ids = array_column($teacher_classes, 'id');
            $placeholders = implode(',', array_fill(0, count($class_ids), '?'));
            $conditions[] = "class_id IN ($placeholders)";
            $params = array_merge($params, $class_ids);
            $types .= str_repeat('i', count($class_ids));
        }
        
        if (!empty($teacher_sections)) {
            $section_ids = array_column($teacher_sections, 'id');
            $placeholders = implode(',', array_fill(0, count($section_ids), '?'));
            $conditions[] = "section_id IN ($placeholders)";
            $params = array_merge($params, $section_ids);
            $types .= str_repeat('i', count($section_ids));
        }
        
        $attendance_day_query .= " AND (" . implode(" OR ", $conditions) . ")";
    }
    
    $attendance_day_stmt = $conn->prepare($attendance_day_query);
    if (!empty($params)) {
        $attendance_day_stmt->bind_param($types, ...$params);
    }
    $attendance_day_stmt->execute();
    $attendance_day_result = $attendance_day_stmt->get_result();
    $attendance_day = $attendance_day_result->fetch_assoc();
    
    $present_counts[] = $attendance_day['present_count'] + ($attendance_day['late_count'] * 0.5) + ($attendance_day['half_day_count'] * 0.5);
    $absent_counts[] = $attendance_day['absent_count'] + ($attendance_day['late_count'] * 0.5) + ($attendance_day['half_day_count'] * 0.5);
    
    $attendance_day_stmt->close();
}

$attendance_chart_data = [
    'dates' => $attendance_dates,
    'present' => $present_counts,
    'absent' => $absent_counts
];


?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Dashboard - School Management System</title>
    
    <!-- Chart.js for charts -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Chart.js datalabels plugin -->
    <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2.0.0"></script>
    
    <style>
        .dashboard-card {
            transition: all 0.3s ease;
           
            overflow: hidden;
            background: white;
        }
        
        .dashboard-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }
        
        .stat-card {
            background: white;
           
            padding: 1.5rem;
            position: relative;
          
        }
        
        .stat-card.students { border-left-color: #3b82f6; }
        .stat-card.classes { border-left-color: #10b981; }
        .stat-card.sections { border-left-color: #8b5cf6; }
        .stat-card.attendance { border-left-color: #f59e0b; }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
        }
        
        .stat-icon.students { background: #dbeafe; color: #3b82f6; }
        .stat-icon.classes { background: #dcfce7; color: #10b981; }
        .stat-icon.sections { background: #f3e8ff; color: #8b5cf6; }
        .stat-icon.attendance { background: #fef3c7; color: #f59e0b; }
        
        .gradient-text {
            background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .attendance-badge {
            padding: 0.25rem 0.5rem;
            border-radius: 0.375rem;
            font-size: 0.75rem;
            font-weight: 500;
        }
        
        .badge-present { background: #dcfce7; color: #166534; }
        .badge-absent { background: #fee2e2; color: #991b1b; }
        .badge-late { background: #fef3c7; color: #92400e; }
        .badge-half-day { background: #dbeafe; color: #1e40af; }
        
        .quick-action-card {
            transition: all 0.2s ease;
            border: 1px solid #e5e7eb;
            border-radius: 0.5rem;
            padding: 1rem;
            text-align: center;
        }
        
        .quick-action-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            border-color: #3b82f6;
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        
        .mini-chart-container {
            position: relative;
            height: 200px;
            width: 100%;
        }
        
        .progress-bar {
            height: 8px;
            border-radius: 4px;
            background: #e5e7eb;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            border-radius: 4px;
            transition: width 0.3s ease;
        }
        
        .quiz-score-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 0.75rem;
            padding: 1rem;
        }
    </style>
</head>
<body class="bg-gray-50">
    <div class="container mx-auto px-2 py-2">
        <!-- Header Section -->
        <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-8">
            <div>
                <h1 class="text-2xl md:text-3xl font-bold text-gray-800">Teacher Dashboard</h1>
                <p class="text-gray-600 mt-1">Welcome back, <?php echo $teacher['full_name']; ?>!</p>
            </div>
            <div class="flex items-center space-x-4 mt-4 md:mt-0">
                <div class="text-right">
                    <p class="text-sm text-gray-600">Teacher ID: <?php echo $teacher['teacher_id']; ?></p>
                    <p class="text-sm text-gray-600"><?php echo date('l, F j, Y'); ?></p>
                </div>
                <?php if (!empty($teacher['photo'])): ?>
                    <img src="../uploads/profiles/<?php echo $teacher['photo']; ?>" 
                         alt="Profile" class="w-10 h-10 rounded-full border-2 border-white shadow-sm">
                <?php else: ?>
                    <div class="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center text-white font-bold shadow-sm">
                        <?php echo substr($teacher['full_name'], 0, 1); ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Stats Overview -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
            <div class="stat-card students">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-2xl font-bold text-gray-800"><?php echo $total_students; ?></h3>
                        <p class="text-gray-600 text-sm">Total Students</p>
                    </div>
                    <div class="stat-icon students">
                        <i class="fas fa-user-graduate"></i>
                    </div>
                </div>
            </div>
            
          
            
            <div class="stat-card sections">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-2xl "><?php echo count($teacher_sections); ?></h3>
                        <p class="text-gray-600 text-sm">Class Assigned</p>
                    </div>
                    <div class="stat-icon sections">
                        <i class="fas fa-users"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card attendance">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-2xl font-bold text-gray-800">
                            <?php echo $attendance_summary['present'] + $attendance_summary['late'] + ($attendance_summary['half_day'] * 0.5); ?>
                        </h3>
                        <p class="text-gray-600 text-sm">Today's Attendance</p>
                    </div>
                    <div class="stat-icon attendance">
                        <i class="fas fa-calendar-check"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Content Grid -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Left Column -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Performance Charts -->
                <div class="dashboard-card p-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-6 gradient-text">Performance Analytics</h2>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Quiz Performance Chart -->
                        <div>
                            <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                                <i class="fas fa-chart-bar mr-2 text-blue-600"></i>
                                Quiz Performance
                            </h3>
                            <div class="chart-container">
                                <canvas id="quizPerformanceChart"></canvas>
                            </div>
                        </div>
                        
                        <!-- Attendance Trend Chart -->
                        <div>
                            <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                                <i class="fas fa-calendar-alt mr-2 text-green-600"></i>
                                Attendance Trend
                            </h3>
                            <div class="chart-container">
                                <canvas id="attendanceTrendChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Quiz Performance Details -->
                <?php if (!empty($quiz_performance_data)): ?>
                <div class="dashboard-card p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Quiz Performance Details</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        <?php foreach ($quiz_performance_data as $quiz): ?>
                            <div class="quiz-score-card">
                                <h4 class="font-semibold mb-2"><?php echo htmlspecialchars($quiz['quiz_title']); ?></h4>
                                <div class="flex justify-between items-center mb-2">
                                    <span class="text-sm">Average Score:</span>
                                    <span class="font-bold text-lg"><?php echo number_format($quiz['avg_score'], 1); ?>%</span>
                                </div>
                                <div class="flex justify-between items-center text-xs">
                                    <span>Attempts: <?php echo $quiz['attempts']; ?></span>
                                    <span>Range: <?php echo number_format($quiz['min_score'], 1); ?>-<?php echo number_format($quiz['max_score'], 1); ?>%</span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Recent Announcements -->
                
            </div>

            <!-- Right Column -->
            <div class="space-y-6">
                <!-- Quick Actions -->
                <div class="dashboard-card p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Quick Actions</h3>
                    <div class="grid grid-cols-2 gap-3">
                        <a href="attendance.php" class="quick-action-card bg-blue-50 hover:bg-blue-100 border-blue-200">
                            <div class="text-blue-600 text-2xl mb-2">
                                <i class="fas fa-calendar-check"></i>
                            </div>
                            <p class="text-sm font-medium text-blue-800">Take Attendance</p>
                        </a>
                        <a href="student.php" class="quick-action-card bg-green-50 hover:bg-green-100 border-green-200">
                            <div class="text-green-600 text-2xl mb-2">
                                <i class="fas fa-user-graduate"></i>
                            </div>
                            <p class="text-sm font-medium text-green-800">View Students</p>
                        </a>
                        <a href="teacher_quiz.php" class="quick-action-card bg-purple-50 hover:bg-purple-100 border-purple-200">
                            <div class="text-purple-600 text-2xl mb-2">
                                <i class="fas fa-tasks"></i>
                            </div>
                            <p class="text-sm font-medium text-purple-800">Create Quiz</p>
                        </a>
                        <a href="report.php" class="quick-action-card bg-orange-50 hover:bg-orange-100 border-orange-200">
                            <div class="text-orange-600 text-2xl mb-2">
                                <i class="fas fa-chart-pie"></i>
                            </div>
                            <p class="text-sm font-medium text-orange-800">Generate Reports</p>
                        </a>
                    </div>
                </div>

                <!-- Today's Attendance Summary -->
                <div class="dashboard-card p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Today's Attendance</h3>
                    <div class="grid grid-cols-2 gap-4 mb-4">
                        <div class="text-center p-3 bg-green-50 rounded-lg">
                            <div class="text-2xl font-bold text-green-600"><?php echo $attendance_summary['present']; ?></div>
                            <div class="text-sm text-green-800">Present</div>
                        </div>
                        <div class="text-center p-3 bg-red-50 rounded-lg">
                            <div class="text-2xl font-bold text-red-600"><?php echo $attendance_summary['absent']; ?></div>
                            <div class="text-sm text-red-800">Absent</div>
                        </div>
                        <div class="text-center p-3 bg-yellow-50 rounded-lg">
                            <div class="text-2xl font-bold text-yellow-600"><?php echo $attendance_summary['late']; ?></div>
                            <div class="text-sm text-yellow-800">Late</div>
                        </div>
                        <div class="text-center p-3 bg-blue-50 rounded-lg">
                            <div class="text-2xl font-bold text-blue-600"><?php echo $attendance_summary['half_day']; ?></div>
                            <div class="text-sm text-blue-800">Half Day</div>
                        </div>
                    </div>
                    <div class="text-center">
                        <a href="attendance.php" class="text-blue-600 hover:text-blue-800 text-sm inline-flex items-center">
                            View Detailed Report <i class="fas fa-arrow-right ml-1 text-xs"></i>
                        </a>
                    </div>
                </div>

                <!-- Leave Balance -->
           
                <!-- Upcoming Events -->
                <div class="dashboard-card p-6">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-lg font-semibold text-gray-800">Upcoming Events</h3>
                        <a href="events.php" class="text-blue-600 hover:text-blue-800 text-sm flex items-center">
                            View All <i class="fas fa-arrow-right ml-1 text-xs"></i>
                        </a>
                    </div>
                   <div class="space-y-3">
    <?php if (!empty($upcoming_events)): ?>
        <?php foreach ($upcoming_events as $event): ?>
            <div class="p-3 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
                <div class="flex items-start space-x-3">
                    <div class="flex-shrink-0 w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                        <span class="text-blue-600 font-bold text-sm">
                            <?php echo date('d', strtotime($event['event_date'])); ?>
                        </span>
                    </div>
                    <div class="flex-1 min-w-0">
                        <h4 class="text-sm font-medium text-gray-800 truncate">
                            <?php echo htmlspecialchars($event['title']); ?>
                        </h4>
                        <p class="text-xs text-gray-500 mt-1">
                            <?php echo date('g:i A', strtotime($event['start_time'])); ?>
                        </p>
                        <p class="text-xs text-gray-500">
                            <?php echo htmlspecialchars($event['location']); ?>
                        </p>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    <?php else: ?>
        <p class="text-gray-500 text-center py-4">No upcoming events</p>
    <?php endif; ?>
</div>

                </div>
            </div>
        </div>
    </div>

    <!-- JavaScript for Charts -->
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Register the datalabels plugin
        Chart.register(ChartDataLabels);
        
        // Initialize Chart.js charts
        initCharts();
    });

    function initCharts() {
        // Quiz Performance Chart
        const quizCtx = document.getElementById('quizPerformanceChart').getContext('2d');
        new Chart(quizCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode(array_column($quiz_performance_data, 'quiz_title')); ?>,
                datasets: [{
                    label: 'Average Score (%)',
                    data: <?php echo json_encode(array_column($quiz_performance_data, 'avg_score')); ?>,
                    backgroundColor: [
                        'rgba(59, 130, 246, 0.8)',
                        'rgba(16, 185, 129, 0.8)',
                        'rgba(139, 92, 246, 0.8)',
                        'rgba(245, 158, 11, 0.8)',
                        'rgba(236, 72, 153, 0.8)'
                    ],
                    borderColor: [
                        'rgb(59, 130, 246)',
                        'rgb(16, 185, 129)',
                        'rgb(139, 92, 246)',
                        'rgb(245, 158, 11)',
                        'rgb(236, 72, 153)'
                    ],
                    borderWidth: 1,
                    borderRadius: 6
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        ticks: {
                            callback: function(value) {
                                return value + '%';
                            }
                        },
                        grid: {
                            drawBorder: false
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return `Score: ${context.raw}%`;
                            }
                        }
                    },
                    datalabels: {
                        anchor: 'end',
                        align: 'top',
                        formatter: function(value) {
                            return value + '%';
                        },
                        color: '#374151',
                        font: {
                            weight: 'bold'
                        }
                    }
                }
            }
        });

        // Attendance Trend Chart
        const attendanceCtx = document.getElementById('attendanceTrendChart').getContext('2d');
        new Chart(attendanceCtx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($attendance_chart_data['dates']); ?>,
                datasets: [
                    {
                        label: 'Present',
                        data: <?php echo json_encode($attendance_chart_data['present']); ?>,
                        fill: true,
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        borderColor: 'rgb(16, 185, 129)',
                        tension: 0.4,
                        pointBackgroundColor: 'rgb(16, 185, 129)',
                        pointBorderColor: '#fff',
                        pointHoverBackgroundColor: '#fff',
                        pointHoverBorderColor: 'rgb(16, 185, 129)',
                        pointRadius: 4,
                        pointHoverRadius: 6
                    },
                    {
                        label: 'Absent',
                        data: <?php echo json_encode($attendance_chart_data['absent']); ?>,
                        fill: true,
                        backgroundColor: 'rgba(239, 68, 68, 0.1)',
                        borderColor: 'rgb(239, 68, 68)',
                        tension: 0.4,
                        pointBackgroundColor: 'rgb(239, 68, 68)',
                        pointBorderColor: '#fff',
                        pointHoverBackgroundColor: '#fff',
                        pointHoverBorderColor: 'rgb(239, 68, 68)',
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            drawBorder: false
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                },
                plugins: {
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                    },
                    legend: {
                        position: 'top',
                    }
                }
            }
        });
    }
    </script>
</body>
</html>

<?php
include 'include/footer.php';

// Close database connection
$conn->close();
?>