<?php
session_start();
require '../config/conn.php';

if (!isset($_SESSION['user_id'])) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        header('Content-Type: application/json');
        echo json_encode(['success'=>false, 'message'=>'Not logged in']);
        exit;
    } else {
        header("Location: login.php");
        exit();
    }
}

$userId = $_SESSION['user_id'];

// Handle AJAX POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');

    $oldPassword = $_POST['old_password'] ?? '';
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';

    // Get current password hash
    $stmt = $conn->prepare("SELECT password FROM users WHERE id=?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $stmt->bind_result($currentHash);
    $stmt->fetch();
    $stmt->close();

    if (!password_verify($oldPassword, $currentHash)) {
        echo json_encode(['success'=>false, 'message'=>'Old password is incorrect.']);
        exit;
    }
    if ($newPassword !== $confirmPassword) {
        echo json_encode(['success'=>false, 'message'=>'New passwords do not match.']);
        exit;
    }

    $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
    $update = $conn->prepare("UPDATE users SET password=? WHERE id=?");
    $update->bind_param("si", $newHash, $userId);
    if ($update->execute()) {
        echo json_encode(['success'=>true, 'message'=>'Password changed successfully!']);
    } else {
        echo json_encode(['success'=>false, 'message'=>'Failed to update password: '.$conn->error]);
    }
    exit;
}

include 'include/header.php';
?>

<div class="p-6 max-w-lg mx-auto">
  <h2 class="text-2xl font-bold mb-4">Change Password</h2>

  <form id="changePasswordForm" class="space-y-4 bg-white p-6 rounded-lg shadow">
    <div>
      <label class="block text-sm font-medium">Old Password</label>
      <input type="password" name="old_password" class="w-full border rounded px-3 py-2" required>
    </div>
    <div>
      <label class="block text-sm font-medium">New Password</label>
      <input type="password" name="new_password" class="w-full border rounded px-3 py-2" required>
    </div>
    <div>
      <label class="block text-sm font-medium">Confirm New Password</label>
      <input type="password" name="confirm_password" class="w-full border rounded px-3 py-2" required>
    </div>
    <button type="submit" class="bg-primary text-white px-4 py-2 rounded hover:bg-primaryDark">
      Update Password
    </button>
    <a href="profile.php" class="ml-2 text-gray-500">Cancel</a>
  </form>
</div>

<script>
const form = document.getElementById('changePasswordForm');

form.addEventListener('submit', function(e){
    e.preventDefault();
    const formData = new FormData(form);

    fetch('change-password.php', {method:'POST', body:formData})
    .then(res => res.json())
    .then(data => {
        showToast(data.message, data.success ? 'success' : 'error');
        if(data.success){
            form.reset();
        }
    })
    .catch(()=>showToast('Something went wrong. Please try again.', 'error'));
});

function showToast(message, type='success'){
    let toast = document.getElementById('toast');
    if(!toast){
        toast = document.createElement('div');
        toast.id = 'toast';
        toast.className = 'toast fixed top-4 right-4 px-4 py-2 rounded shadow text-white z-50';
        document.body.appendChild(toast);
    }
    toast.textContent = message;
    toast.className = 'toast fixed top-4 right-4 px-4 py-2 rounded shadow text-white z-50 ' + (type==='success'?'bg-green-500':'bg-red-500');
    toast.style.display = 'block';
    setTimeout(()=>{ toast.style.display='none'; }, 4000);
}
</script>

<?php include 'include/footer.php'; ?>
