<?php
// Enable output buffering
ob_start();

// Database connection details
require 'config/conn.php';

// Initialize variables
$errors = [];
$success = false;
$name = $email = '';

// Check if form was submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get form data
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $terms = isset($_POST['terms']) ? true : false;
    $role = 'admin'; // Force role to be admin only

    // Validation
    if (empty($name)) {
        $errors[] = "Full name is required";
    } elseif (strlen($name) < 3) {
        $errors[] = "Name must be at least 3 characters";
    }
    
    if (empty($email)) {
        $errors[] = "Email is required";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Valid email is required";
    }
    
    if (empty($password)) {
        $errors[] = "Password is required";
    } elseif (strlen($password) < 8) {
        $errors[] = "Password must be at least 8 characters";
    } elseif (!preg_match('/[0-9]/', $password)) {
        $errors[] = "Password must contain at least one number";
    }
    
    if ($password !== $confirm_password) {
        $errors[] = "Passwords do not match";
    }
    
    if (!$terms) {
        $errors[] = "You must agree to the terms and conditions";
    }

    // If no errors, proceed with registration
    if (empty($errors)) {
        try {
            // Create database connection
            $conn = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
            $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // Check if email exists
            $check_email = $conn->prepare("SELECT * FROM users WHERE email = ?");
            $check_email->execute([$email]);
            
            if ($check_email->rowCount() > 0) {
                $errors[] = "Email already exists";
            } else {
                // Hash password
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $current_time = date('Y-m-d H:i:s');
                
                // Insert user
                $query = "INSERT INTO users (name, email, password, role, school, image, created_at, password_updated_at, profile_completion) 
                          VALUES (?, ?, ?, ?, NULL, NULL, ?, ?, 0)";
                
                $stmt = $conn->prepare($query);
                if ($stmt->execute([$name, $email, $hashed_password, $role, $current_time, $current_time])) {
                    $success = true;
                    // Clear form fields
                    $name = $email = '';
                } else {
                    $errors[] = "Registration failed. Please try again.";
                }
            }
        } catch(PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>EduManage Pro - Registration</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script>
    tailwind.config = {
      theme: {
        extend: {
          colors: {
            primary: '#0096FF',
            secondary: '#0077CC',
            accent: '#FF6B6B',
            dark: '#1E293B',
            light: '#F8FAFC'
          },
          fontFamily: {
            sans: ['Inter', 'sans-serif'],
          },
        }
      }
    }
  </script>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <style>
    .auth-container {
      background: linear-gradient(135deg, #0096FF 0%, #0077CC 100%);
    }
    .auth-card {
      box-shadow: 0 10px 30px rgba(0, 150, 255, 0.2);
    }
    .input-field:focus {
      box-shadow: 0 0 0 3px rgba(0, 150, 255, 0.2);
    }
    
    /* Toast notification styles */
    .toast {
      position: fixed;
      top: 20px;
      right: 20px;
      padding: 16px 24px;
      border-radius: 8px;
      color: white;
      font-weight: 500;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
      z-index: 1000;
      transform: translateX(150%);
      transition: transform 0.3s ease-in-out;
      display: flex;
      align-items: center;
      max-width: 350px;
    }
    .toast.show {
      transform: translateX(0);
    }
    .toast.success {
      background-color: #10B981;
    }
    .toast.error {
      background-color: #EF4444;
    }
    .toast.warning {
      background-color: #F59E0B;
    }
    .toast.info {
      background-color: #3B82F6;
    }
    .toast i {
      margin-right: 12px;
      font-size: 20px;
    }
    @media (max-width: 640px) {
      .toast {
        top: 10px;
        right: 10px;
        left: 10px;
        max-width: calc(100% - 20px);
      }
    }
  </style>
</head>
<body class="font-sans bg-gray-50">

<!-- Toast Notification -->
<div id="toast" class="toast hidden">
  <i id="toastIcon"></i>
  <span id="toastMessage"></span>
</div>

<!-- Register Page -->
<div class="min-h-screen flex items-center justify-center auth-container p-4">
  <div class="auth-card bg-white rounded-xl p-8 w-full max-w-md">
    <div class="text-center mb-8">
      <div class="flex justify-center mb-4">
        <i class="fas fa-graduation-cap text-primary text-4xl"></i>
      </div>
      <h1 class="text-2xl font-bold text-gray-800">Create Account</h1>
      <p class="text-gray-600 mt-2">Setup your admin credentials</p>
    </div>

    <?php if (!empty($errors)): ?>
      <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-4" role="alert">
        <p class="font-bold">Registration Failed</p>
        <?php foreach ($errors as $error): ?>
          <p><?php echo htmlspecialchars($error); ?></p>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>

    <?php if ($success): ?>
      <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-4" role="alert">
        <p class="font-bold">Registration Successful</p>
        <p>Your account has been created. You can now <a href="login.php" class="font-medium underline">login here</a>.</p>
      </div>
    <?php endif; ?>

    <form id="registerForm" method="POST" action="">
      <div class="mb-4">
        <label for="regName" class="block text-sm font-medium text-gray-700 mb-1">Full Name</label>
        <input type="text" id="regName" name="name" class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary" placeholder="John Doe" value="<?php echo htmlspecialchars($name); ?>" required>
      </div>

      <div class="mb-4">
        <label for="regEmail" class="block text-sm font-medium text-gray-700 mb-1">Email Address</label>
        <input type="email" id="regEmail" name="email" class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary" placeholder="admin@school.edu" value="<?php echo htmlspecialchars($email); ?>" required>
      </div>

      <div class="mb-4">
        <label for="regPassword" class="block text-sm font-medium text-gray-700 mb-1">Password</label>
        <div class="relative">
          <input type="password" id="regPassword" name="password" class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary" placeholder="••••••••" required>
          <button type="button" class="absolute right-3 top-3 text-gray-400 hover:text-gray-600" onclick="togglePassword('regPassword')">
            <i class="far fa-eye"></i>
          </button>
        </div>
        <p class="text-xs text-gray-500 mt-1">Minimum 8 characters with at least one number</p>
      </div>

      <div class="mb-6">
        <label for="regConfirmPassword" class="block text-sm font-medium text-gray-700 mb-1">Confirm Password</label>
        <div class="relative">
          <input type="password" id="regConfirmPassword" name="confirm_password" class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary" placeholder="••••••••" required>
          <button type="button" class="absolute right-3 top-3 text-gray-400 hover:text-gray-600" onclick="togglePassword('regConfirmPassword')">
            <i class="far fa-eye"></i>
          </button>
        </div>
      </div>

      <div class="mb-6">
        <input type="checkbox" id="terms" name="terms" class="h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded" required>
        <label for="terms" class="ml-2 block text-sm text-gray-700">I agree to the <a href="#" class="text-primary hover:text-secondary">Terms of Service</a> and <a href="#" class="text-primary hover:text-secondary">Privacy Policy</a></label>
      </div>

      <button type="submit" class="w-full bg-primary hover:bg-secondary text-white font-medium py-2 px-4 rounded-lg transition duration-200 mb-4">
        Create Account
      </button>
    </form>

    <div class="mt-4 text-center">
      <p class="text-sm text-gray-600">Already have an account? <a href="login.php" class="text-primary hover:text-secondary font-medium">Sign in here</a></p>
    </div>
  </div>
</div>

<script>
  // Toast notification function
  function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    const toastIcon = document.getElementById('toastIcon');
    const toastMessage = document.getElementById('toastMessage');
    
    // Set toast content and style
    toastMessage.textContent = message;
    toast.className = 'toast'; // Reset classes
    toast.classList.add(type);
    
    // Set icon based on type
    switch(type) {
      case 'success':
        toastIcon.className = 'fas fa-check-circle';
        break;
      case 'error':
        toastIcon.className = 'fas fa-exclamation-circle';
        break;
      case 'warning':
        toastIcon.className = 'fas fa-exclamation-triangle';
        break;
      default:
        toastIcon.className = 'fas fa-info-circle';
    }
    
    // Show toast
    toast.classList.remove('hidden');
    toast.classList.add('show');
    
    // Hide after 5 seconds
    setTimeout(() => {
      toast.classList.remove('show');
      setTimeout(() => toast.classList.add('hidden'), 300);
    }, 5000);
  }

  // Toggle password visibility
  function togglePassword(fieldId) {
    const field = document.getElementById(fieldId);
    const icon = field.nextElementSibling.querySelector('i');
    
    if (field.type === 'password') {
      field.type = 'text';
      icon.classList.remove('fa-eye');
      icon.classList.add('fa-eye-slash');
    } else {
      field.type = 'password';
      icon.classList.remove('fa-eye-slash');
      icon.classList.add('fa-eye');
    }
  }

  // Form validation
  document.getElementById('registerForm').addEventListener('submit', function(e) {
    const password = document.getElementById('regPassword').value;
    const confirmPassword = document.getElementById('regConfirmPassword').value;
    const termsChecked = document.getElementById('terms').checked;
    
    if (!termsChecked) {
      e.preventDefault();
      showToast('You must agree to the terms and conditions', 'error');
      return;
    }
    
    if (password !== confirmPassword) {
      e.preventDefault();
      showToast('Passwords do not match', 'error');
      return;
    }
    
    if (password.length < 8) {
      e.preventDefault();
      showToast('Password must be at least 8 characters', 'error');
      return;
    }
    
    if (!/\d/.test(password)) {
      e.preventDefault();
      showToast('Password must contain at least one number', 'error');
      return;
    }
    
    // Show loading state
    const submitBtn = this.querySelector('button[type="submit"]');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Creating Account...';
  });

  // Show toast messages from PHP
  <?php if (!empty($errors)): ?>
    <?php foreach ($errors as $error): ?>
      showToast('<?php echo addslashes($error); ?>', 'error');
    <?php endforeach; ?>
  <?php endif; ?>

  <?php if ($success): ?>
    showToast('Registration successful! You can now login.', 'success');
  <?php endif; ?>
</script>
</body>
</html>
<?php
// Flush output buffer
ob_end_flush();
?>