<?php
// Enable output buffering at the very start
ob_start();

// Custom error handling for session issues
set_error_handler(function($severity, $message, $file, $line) {
    if (strpos($message, 'session') !== false) {
        error_log("Session error: " . $message);
    }
});

// Include database connection
include 'config/conn.php';

// Enhanced session configuration
$sessionName = 'ADINOHEIGHT_SESSION';
$sessionLifetime = 0; // Until browser closes
$sessionPath = '/';
$sessionSecure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
$sessionHttpOnly = true;
$sessionSameSite = 'Lax';

// Set session name to avoid conflicts
session_name($sessionName);

// Start session with error handling
try {
    if (!session_start()) {
        throw new Exception('Failed to start session');
    }
    
    // Regenerate session ID to prevent fixation attacks
    if (empty($_SESSION['initiated'])) {
        session_regenerate_id(true);
        $_SESSION['initiated'] = true;
    }
} catch (Exception $e) {
    error_log("Session start failed: " . $e->getMessage());
    $sessionFailed = true;
}

// Check if session is working
if (session_status() !== PHP_SESSION_ACTIVE) {
    $sessionFailed = true;
    $error = 'Session initialization failed. Please check your browser settings.';
}

// Initialize variables
$error = $error ?? '';
$username = '';
$user_type = 'student';

// Process login if form was submitted
if (!$sessionFailed && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $user_type = $_POST['user_type'] ?? 'student';

    if (empty($username) || empty($password)) {
        $error = 'Username/ID and password are required';
    } else {
        // ---------------- STUDENT LOGIN ----------------
        if ($user_type === 'student') {
            $query = "SELECT id, student_id, username, password, full_name, email, status 
                      FROM students 
                      WHERE username = ? OR student_id = ?";
            $stmt = $conn->prepare($query);

            if ($stmt) {
                $stmt->bind_param("ss", $username, $username);
                $stmt->execute();
                $result = $stmt->get_result();

                if ($result->num_rows === 1) {
                    $user = $result->fetch_assoc();

                    if (password_verify($password, $user['password'])) {
                        if ($user['status'] !== 'active') {
                            $error = 'Your account is not active. Please contact administration.';
                        } else {
                            // Clear any existing session data
                            $_SESSION = array();
                            
                            // Set session data
                            $_SESSION['user_id'] = $user['id'];
                            $_SESSION['username'] = $user['username'];
                            $_SESSION['full_name'] = $user['full_name'];
                            $_SESSION['email'] = $user['email'];
                            $_SESSION['user_type'] = 'student';
                            $_SESSION['last_activity'] = time();
                            $_SESSION['logged_in'] = true;
                            $_SESSION['session_created'] = time();

                            // Write session immediately
                            session_write_close();

                            header('Location: student/index.php');
                            exit();
                        }
                    } else {
                        $error = 'Invalid password';
                    }
                } else {
                    $error = 'Student account not found';
                }
                $stmt->close();
            } else {
                $error = 'Database error: ' . $conn->error;
            }

        // ---------------- TEACHER LOGIN (from users table) ----------------
        } else if ($user_type === 'teacher') {
            $query = "SELECT id, teacher_id, username, password, full_name, email, status, role 
                      FROM users 
                      WHERE (username = ? OR teacher_id = ?) AND role = 'teacher'";
            $stmt = $conn->prepare($query);

            if ($stmt) {
                $stmt->bind_param("ss", $username, $username);
                $stmt->execute();
                $result = $stmt->get_result();

                if ($result->num_rows === 1) {
                    $user = $result->fetch_assoc();

                    if (password_verify($password, $user['password'])) {
                        if ($user['status'] !== 'active') {
                            $error = 'Your account is not active. Please contact administration.';
                        } else {
                            // Clear any existing session data
                            $_SESSION = array();
                            
                            $_SESSION['user_id'] = $user['id'];
                            $_SESSION['username'] = $user['username'];
                            $_SESSION['full_name'] = $user['full_name'];
                            $_SESSION['email'] = $user['email'];
                            $_SESSION['user_type'] = 'teacher';
                            $_SESSION['last_activity'] = time();
                            $_SESSION['logged_in'] = true;
                            $_SESSION['session_created'] = time();

                            // Write session immediately
                            session_write_close();

                            header('Location: teacher/index.php');
                            exit();
                        }
                    } else {
                        $error = 'Invalid password';
                    }
                } else {
                    $error = 'Teacher account not found';
                }
                $stmt->close();
            } else {
                $error = 'Database error: ' . $conn->error;
            }

        // ---------------- PARENT LOGIN ----------------
        } else if ($user_type === 'parent') {
            $query = "SELECT id, username, password, fullname, email, contact 
                      FROM parents 
                      WHERE username = ? OR email = ?";
            $stmt = $conn->prepare($query);

            if ($stmt) {
                $stmt->bind_param("ss", $username, $username);
                $stmt->execute();
                $result = $stmt->get_result();

                if ($result->num_rows === 1) {
                    $user = $result->fetch_assoc();

                    if (password_verify($password, $user['password'])) {
                        // Clear any existing session data
                        $_SESSION = array();
                        
                        $_SESSION['user_id'] = $user['id'];
                        $_SESSION['username'] = $user['username'];
                        $_SESSION['full_name'] = $user['fullname'];
                        $_SESSION['email'] = $user['email'];
                        $_SESSION['contact'] = $user['contact'];
                        $_SESSION['user_type'] = 'parent';
                        $_SESSION['last_activity'] = time();
                        $_SESSION['logged_in'] = true;
                        $_SESSION['session_created'] = time();

                        // Write session immediately
                        session_write_close();

                        header('Location: parent/index.php');
                        exit();
                    } else {
                        $error = 'Invalid password';
                    }
                } else {
                    $error = 'Parent account not found';
                }
                $stmt->close();
            } else {
                $error = 'Database error: ' . $conn->error;
            }
        }
    }
}

// Close connection at end only if it's still open
if (isset($conn) && $conn !== null) {
    $conn->close();
}
ob_end_flush();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Adinoheight College - Login Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#eff6ff',
                            100: '#dbeafe',
                            200: '#bfdbfe',
                            300: '#93c5fd',
                            400: '#60a5fa',
                            500: '#3b82f6',
                            600: '#2563eb',
                            700: '#1d4ed8',
                            800: '#1e40af',
                            900: '#1e3a8a',
                        },
                        secondary: '#f59e0b',
                    }
                }
            }
        }
    </script>
    <style>
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 8px;
            color: white;
            display: flex;
            align-items: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            z-index: 1000;
            transform: translateX(100%);
            opacity: 0;
            transition: transform 0.3s ease, opacity 0.3s ease;
        }
        .toast.show {
            transform: translateX(0);
            opacity: 1;
        }
        .toast.hidden {
            display: none;
        }
        .toast.success {
            background-color: #10B981;
        }
        .toast.error {
            background-color: #EF4444;
        }
        .toast.warning {
            background-color: #F59E0B;
        }
        .toast i {
            margin-right: 10px;
            font-size: 1.2rem;
        }
        
        /* Debug panel styling */
        .debug-panel {
            position: fixed;
            bottom: 10px;
            right: 10px;
            background: rgba(0,0,0,0.8);
            color: white;
            padding: 10px;
            border-radius: 5px;
            font-size: 12px;
            z-index: 9999;
            max-width: 300px;
            max-height: 200px;
            overflow: auto;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-primary-50 to-primary-100 min-h-screen flex items-center justify-center p-4">
    <!-- Toast Notification -->
    <div id="toast" class="toast hidden">
        <i id="toastIcon"></i>
        <span id="toastMessage"></span>
    </div>

    <!-- Debug Panel (visible only during development) -->
    <?php if (isset($_GET['debug'])): ?>
    <div class="debug-panel">
        <strong>Session Debug Info:</strong><br>
        Status: <?php echo session_status(); ?><br>
        ID: <?php echo session_id(); ?><br>
        Name: <?php echo session_name(); ?><br>
        Cookie Params: <?php print_r(session_get_cookie_params()); ?><br>
        Session Data: <?php print_r($_SESSION); ?>
    </div>
    <?php endif; ?>

    <div class="w-full max-w-md bg-white  overflow-hidden">
        <!-- Header -->
        <div class="bg-gradient-to-r from-primary-700 to-primary-600 text-white p-8 text-center">
            <div class="w-20 h-20 bg-white rounded-full flex items-center justify-center mx-auto mb-4 ">
                 
                            <img src="img/logotwo.png" alt="Adinoheight College Logo">
                        
            </div>
            <h1 class="text-2xl font-bold">Adinoheight College</h1>
            <p class="opacity-90 mt-2">Login to access your portal</p>
        </div>
        
        <!-- Login Form -->
        <div class="p-8">
          <form id="loginForm" method="POST" action="portal_login<?php echo isset($_GET['debug']) ? '?debug=1' : ''; ?>">
                <!-- User Type Selection -->
                <input type="hidden" name="user_type" id="userType" value="<?php echo htmlspecialchars($user_type); ?>">
                
                <div class="flex rounded-lg overflow-hidden border border-gray-300 mb-6">
                    <button type="button" class="user-type-btn flex-1 py-3 px-4 text-center font-medium transition-colors <?php echo $user_type === 'student' ? 'bg-primary-600 text-white' : 'bg-gray-100 text-gray-700'; ?>" data-type="student">
                        Student
                    </button>
                    <button type="button" class="user-type-btn flex-1 py-3 px-4 text-center font-medium transition-colors <?php echo $user_type === 'teacher' ? 'bg-primary-600 text-white' : 'bg-gray-100 text-gray-700'; ?>" data-type="teacher">
                        Teacher
                    </button>
                    <button type="button" class="user-type-btn flex-1 py-3 px-4 text-center font-medium transition-colors <?php echo $user_type === 'parent' ? 'bg-primary-600 text-white' : 'bg-gray-100 text-gray-700'; ?>" data-type="parent">
                        Parent
                    </button>
                </div>
                
                <!-- Username/ID Field -->
                <div class="mb-5">
                    <label for="username" class="block text-sm font-medium text-gray-700 mb-1" id="usernameLabel">
                        <?php 
                        if ($user_type === 'student') {
                            echo 'Student ID or Username';
                        } else if ($user_type === 'teacher') {
                            echo 'Teacher ID or Username';
                        } else {
                            echo 'Username or Email';
                        }
                        ?>
                    </label>
                    <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($username); ?>" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition" 
                        placeholder="<?php 
                        if ($user_type === 'student') {
                            echo 'Enter your student ID or username';
                        } else if ($user_type === 'teacher') {
                            echo 'Enter your teacher ID or username';
                        } else {
                            echo 'Enter your username or email';
                        }
                        ?>" required>
                </div>
                
                <!-- Password Field -->
                <div class="mb-5">
                    <label for="password" class="block text-sm font-medium text-gray-700 mb-1">Password</label>
                    <div class="relative">
                        <input type="password" id="password" name="password" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition" placeholder="Enter your password" required>
                        <button type="button" onclick="togglePassword('password')" class="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-500 hover:text-primary-600">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
                
                <!-- Remember Me -->
                <div class="flex items-center mb-6">
                    <input type="checkbox" id="rememberMe" name="remember_me" class="h-4 w-4 text-primary-600 focus:ring-primary-500 border-gray-300 rounded">
                    <label for="rememberMe" class="ml-2 block text-sm text-gray-700">Remember me</label>
                </div>
                
                <!-- Error Message -->
                <?php if (!empty($error)): ?>
                    <div class="mb-4 p-3 bg-red-50 text-red-700 rounded-lg text-sm">
                        <i class="fas fa-exclamation-circle mr-2"></i><?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>
                
                <!-- Session Error Message -->
                <?php if (isset($sessionFailed)): ?>
                    <div class="mb-4 p-3 bg-yellow-50 text-yellow-700 rounded-lg text-sm">
                        <i class="fas fa-exclamation-triangle mr-2"></i>Session issues detected. Please ensure cookies are enabled in your browser.
                    </div>
                <?php endif; ?>
                
                <!-- Submit Button -->
                <button type="submit" class="w-full bg-primary-600 hover:bg-primary-700 text-white font-semibold py-3 px-4 rounded-lg shadow-md transition-all transform hover:-translate-y-0.5 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:ring-opacity-50">
                    Login to Portal
                </button>
            </form>
            
           
        </div>
    </div>

    <script>
        // Toggle between student, teacher and parent login
        document.querySelectorAll('.user-type-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const userType = this.dataset.type;
                document.getElementById('userType').value = userType;
                
                // Update button styles
                document.querySelectorAll('.user-type-btn').forEach(b => {
                    b.classList.remove('bg-primary-600', 'text-white');
                    b.classList.add('bg-gray-100', 'text-gray-700');
                });
                this.classList.remove('bg-gray-100', 'text-gray-700');
                this.classList.add('bg-primary-600', 'text-white');
                
                // Update placeholder text and label
                const usernameInput = document.getElementById('username');
                const usernameLabel = document.getElementById('usernameLabel');
                
                if (userType === 'student') {
                    usernameInput.placeholder = 'Enter your student ID or username';
                    usernameLabel.textContent = 'Student ID or Username';
                } else if (userType === 'teacher') {
                    usernameInput.placeholder = 'Enter your teacher ID or username';
                    usernameLabel.textContent = 'Teacher ID or Username';
                } else {
                    usernameInput.placeholder = 'Enter your username or email';
                    usernameLabel.textContent = 'Username or Email';
                }
            });
        });

        // Toggle password visibility
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = field.nextElementSibling.querySelector('i');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        // Form validation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            const username = document.getElementById('username').value;
            const password = document.getElementById('password').value;
            
            if (!username) {
                e.preventDefault();
                showToast('Username/ID is required', 'error');
                return;
            }
            
            if (!password) {
                e.preventDefault();
                showToast('Password is required', 'error');
                return;
            }
            
            // Show loading state
            const submitBtn = this.querySelector('button[type="submit"]');
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Signing In...';
        });

        // Toast notification function
        function showToast(message, type = 'success') {
            const toast = document.getElementById('toast');
            const toastIcon = document.getElementById('toastIcon');
            const toastMessage = document.getElementById('toastMessage');
            
            // Set toast content and style
            toastMessage.textContent = message;
            toast.className = 'toast'; // Reset classes
            toast.classList.add(type);
            
            // Set icon based on type
            switch(type) {
                case 'success':
                    toastIcon.className = 'fas fa-check-circle';
                    break;
                case 'error':
                    toastIcon.className = 'fas fa-exclamation-circle';
                    break;
                case 'warning':
                    toastIcon.className = 'fas fa-exclamation-triangle';
                    break;
                default:
                    toastIcon.className = 'fas fa-info-circle';
            }
            
            // Show toast
            toast.classList.remove('hidden');
            toast.classList.add('show');
            
            // Hide after 5 seconds
            setTimeout(() => {
                toast.classList.remove('show');
                setTimeout(() => toast.classList.add('hidden'), 300);
            }, 5000);
        }

        // Handle forgot password
        document.getElementById('forgotPassword').addEventListener('click', function(e) {
            e.preventDefault();
            const userType = document.getElementById('userType').value;
            let message = '';
            
            if (userType === 'student') {
                message = 'For students, please contact the administration office to reset your password.';
            } else if (userType === 'teacher') {
                message = 'For teachers, please contact the administration office to reset your password.';
            } else {
                message = 'For parents, please contact the administration office to reset your password.';
            }
            
            alert(message);
        });
    </script>
</body>
</html>