<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// ✅ Ensure parent login
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'parent') {
    header('Location: login.php');
    exit();
}

$parent_id = $_SESSION['user_id'];
$error = '';

// ✅ Get unread notifications count
$notification_count_query = "SELECT COUNT(*) as unread_count FROM parent_notifications WHERE parent_id = $parent_id AND is_read = 0";
$notification_count_result = mysqli_query($conn, $notification_count_query);
$unread_notifications = 0;
if ($notification_count_result) {
    $notification_data = mysqli_fetch_assoc($notification_count_result);
    $unread_notifications = $notification_data['unread_count'];
}

// ✅ Get children linked to this parent
$children_query = "
    SELECT s.id, s.full_name, s.student_id, s.class, s.section, 
           c.class_name, sec.section_name
    FROM students s
    LEFT JOIN classes c ON s.class = c.id
    LEFT JOIN sections sec ON s.section = sec.id
    WHERE s.parent_id = $parent_id
";
$children_result = mysqli_query($conn, $children_query);

// Check if query was successful
if (!$children_result) {
    $error = "Database error: " . mysqli_error($conn);
}

$children = [];
$child_ids = [];
if ($children_result && mysqli_num_rows($children_result) > 0) {
    while ($row = mysqli_fetch_assoc($children_result)) {
        $children[] = $row;
        $child_ids[] = $row['id'];
    }
}

// Initialize summary data
$summary_data = [
    'total_children' => count($children),
    'total_unpaid_invoices' => 0,
    'total_due_amount' => 0,
    'recent_notices' => 0,
    'upcoming_events' => 0,
    'unread_notifications' => $unread_notifications
];

// Get financial summary if children exist
if (!empty($child_ids)) {
    $child_ids_str = implode(',', $child_ids);
    
    // Get unpaid invoices summary
    $invoices_query = "
        SELECT COUNT(*) as total_unpaid, COALESCE(SUM(i.amount - COALESCE(paid.paid_amount, 0)), 0) as total_due
        FROM invoices i
        LEFT JOIN (
            SELECT invoice_id, SUM(amount) as paid_amount 
            FROM payments 
            GROUP BY invoice_id
        ) paid ON i.id = paid.invoice_id
        WHERE i.student_id IN ($child_ids_str)
        AND (i.amount - COALESCE(paid.paid_amount, 0)) > 0
    ";
    
    $invoices_result = mysqli_query($conn, $invoices_query);
    if ($invoices_result) {
        $invoice_data = mysqli_fetch_assoc($invoices_result);
        $summary_data['total_unpaid_invoices'] = $invoice_data['total_unpaid'];
        $summary_data['total_due_amount'] = $invoice_data['total_due'];
    }
    
    // Get recent notices count (last 7 days)
    $notices_query = "
        SELECT COUNT(*) as recent_count
        FROM notices n
        WHERE (
            n.audience_type = 'all'
            OR (n.audience_type = 'class' AND n.target_id IN (SELECT class FROM students WHERE id IN ($child_ids_str)))
            OR (n.audience_type = 'section' AND n.target_id IN (SELECT section FROM students WHERE id IN ($child_ids_str)))
            OR (n.audience_type = 'student' AND n.target_id IN ($child_ids_str))
        )
        AND n.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        AND (n.expiry_date IS NULL OR n.expiry_date >= CURDATE())
    ";
    
    $notices_result = mysqli_query($conn, $notices_query);
    if ($notices_result) {
        $notice_data = mysqli_fetch_assoc($notices_result);
        $summary_data['recent_notices'] = $notice_data['recent_count'];
    }
    
    // Get upcoming events count (next 30 days)
    $events_query = "
        SELECT COUNT(*) as upcoming_count
        FROM events e
        WHERE e.event_date >= CURDATE() 
        AND e.event_date <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)
        AND (
            e.audience_type = 'all'
            OR (e.audience_type = 'class' AND e.target_id IN (SELECT class FROM students WHERE id IN ($child_ids_str)))
            OR (e.audience_type = 'section' AND e.target_id IN (SELECT section FROM students WHERE id IN ($child_ids_str)))
            OR (e.audience_type = 'student' AND e.target_id IN ($child_ids_str))
        )
    ";
    
    $events_result = mysqli_query($conn, $events_query);
    if ($events_result) {
        $event_data = mysqli_fetch_assoc($events_result);
        $summary_data['upcoming_events'] = $event_data['upcoming_count'];
    }
    
    // Get recent exam results for each child
    foreach ($children as &$child) {
        $child_id = $child['id'];
        
        // Get latest exam result
        $exam_query = "
            SELECT e.exam_name, er.subject_id, s.subject_name, er.ca_score, er.exam_score, 
                   (er.ca_score + er.exam_score) as total_score, er.grade, e.exam_date
            FROM exam_results er
            JOIN exams e ON er.exam_id = e.id
            JOIN subjects s ON er.subject_id = s.id
            WHERE er.student_id = $child_id 
            AND er.is_released = 1
            ORDER BY e.exam_date DESC, s.subject_name
            LIMIT 5
        ";
        
        $exam_result = mysqli_query($conn, $exam_query);
        $child['recent_results'] = [];
        if ($exam_result) {
            while ($exam_row = mysqli_fetch_assoc($exam_result)) {
                $child['recent_results'][] = $exam_row;
            }
        }
        
        // Get attendance summary
        $attendance_query = "
            SELECT 
                COUNT(*) as total_days,
                SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
                SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days,
                SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late_days
            FROM attendance 
            WHERE student_id = $child_id 
            AND attendance_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ";
        
        $attendance_result = mysqli_query($conn, $attendance_query);
        
        // Initialize with default values to prevent undefined key warnings
        $child['attendance'] = [
            'present_days' => 0, 
            'absent_days' => 0, 
            'late_days' => 0,
            'total_days' => 0,
            'attendance_rate' => 0
        ];
        
        if ($attendance_result && mysqli_num_rows($attendance_result) > 0) {
            $attendance_data = mysqli_fetch_assoc($attendance_result);
            $child['attendance'] = array_merge($child['attendance'], $attendance_data);
            
            if ($child['attendance']['total_days'] > 0) {
                $child['attendance']['attendance_rate'] = round(
                    ($child['attendance']['present_days'] / $child['attendance']['total_days']) * 100
                );
            }
        }
    }
    unset($child); // Unset reference
}

ob_end_flush();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Parent Dashboard - Adinoheightschool</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-100">
<div class="container mx-auto px-0 py-0">
    <!-- Welcome Header -->
    <div class="bg-gradient-to-r from-blue-600 to-purple-600 text-white p-2  mb-6">
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold">Welcome, <?php echo $_SESSION['fullname'] ?? 'Parent'; ?>!</h1>
                <p class="opacity-90">Monitor your children's academic progress and school activities</p>
            </div>
            <div class="relative">
                <button id="notificationButton" 
                        class="bg-white text-blue-600 px-4 py-2  font-semibold hover:bg-blue-50 transition-colors flex items-center">
                    <i class="fas fa-bell mr-2"></i>
                    <span id="notificationButtonText">Enable Notifications</span>
                    <span id="notificationBadge" class="notification-badge absolute -top-2 -right-2 bg-red-500 text-white rounded-full w-6 h-6 text-xs flex items-center justify-center hidden">
                        <?php echo $unread_notifications; ?>
                    </span>
                </button>
                
                <button id="refreshNotifications" 
                        class="ml-2 bg-gray-100 text-gray-600 px-3 py-2  hover:bg-gray-200 transition-colors flex items-center text-sm">
                    <i class="fas fa-sync-alt mr-1"></i> Refresh
                </button>
            </div>
        </div>
    </div>

    <?php if (!empty($error)): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
            <?php echo $error; ?>
        </div>
    <?php endif; ?>

    <!-- Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="bg-white p-6  border-l-4 border-blue-500 ">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 rounded-full mr-4">
                    <i class="fas fa-user-graduate text-blue-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['total_children']; ?></h3>
                    <p class="text-gray-600">Children</p>
                </div>
            </div>
        </div>

        <div class="bg-white p-6  border-l-4 border-red-500 ">
            <div class="flex items-center">
                <div class="bg-red-100 p-3 rounded-full mr-4">
                    <i class="fas fa-file-invoice-dollar text-red-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['total_unpaid_invoices']; ?></h3>
                    <p class="text-gray-600">Unpaid Invoices</p>
                    <p class="text-sm text-red-600 font-semibold">
                        ₦<?php echo number_format($summary_data['total_due_amount'], 2); ?> due
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white p-6  border-l-4 border-green-500 ">
            <div class="flex items-center">
                <div class="bg-green-100 p-3 rounded-full mr-4">
                    <i class="fas fa-bell text-green-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['unread_notifications']; ?></h3>
                    <p class="text-gray-600">New Notifications</p>
                </div>
            </div>
        </div>

        <div class="bg-white p-6  border-l-4 border-purple-500 ">
            <div class="flex items-center">
                <div class="bg-purple-100 p-3 rounded-full mr-4">
                    <i class="fas fa-calendar-alt text-purple-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['upcoming_events']; ?></h3>
                    <p class="text-gray-600">Upcoming Events</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Children Overview -->
    <?php if (!empty($children)): ?>
        <div class="bg-white   mb-8">
            <div class="p-6 border-b">
                <h2 class="text-xl font-bold text-gray-800">Your Children</h2>
            </div>
            
            <div class="divide-y">
                <?php foreach ($children as $child): ?>
                    <div class="p-6">
                        <div class="flex justify-between items-start mb-4">
                            <div>
                                <h3 class="text-lg font-semibold text-gray-800"><?php echo htmlspecialchars($child['full_name']); ?></h3>
                                <p class="text-gray-600">
                                    <?php echo htmlspecialchars($child['class_name'] . ' - ' . $child['section_name']); ?> |
                                    ID: <?php echo htmlspecialchars($child['student_id']); ?>
                                </p>
                            </div>
                            <div class="text-right">
                                <span class="px-3 py-1 rounded-full text-xs font-medium 
                                    <?php echo $child['attendance']['attendance_rate'] >= 90 ? 'bg-green-100 text-green-800' : 
                                           ($child['attendance']['attendance_rate'] >= 75 ? 'bg-yellow-100 text-yellow-800' : 
                                           'bg-red-100 text-red-800'); ?>">
                                    Attendance: <?php echo $child['attendance']['attendance_rate']; ?>%
                                </span>
                            </div>
                        </div>


                        <!-- Quick Actions -->
                        <div class="flex space-x-3">
                            <a href="result.php?student_id=<?php echo $child['id']; ?>" 
                               class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded text-sm flex items-center">
                                <i class="fas fa-chart-line mr-1"></i> View Results
                            </a>
                            <a href="attendance.php?student_id=<?php echo $child['id']; ?>" 
                               class="bg-green-600 hover:bg-green-700 text-white px-3 py-1 rounded text-sm flex items-center">
                                <i class="fas fa-calendar-check mr-1"></i> View Attendance
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php else: ?>
        <div class="bg-white p-8   text-center">
            <i class="fas fa-user-graduate text-4xl text-gray-300 mb-4"></i>
            <h3 class="text-lg font-medium text-gray-600">No children linked to your account</h3>
            <p class="text-gray-500">Please contact the school administration to link your children to your parent account.</p>
        </div>
    <?php endif; ?>

    <!-- Quick Links -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        <a href="notices.php" class="bg-white p-6   hover:shadow-md transition-shadow">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 rounded-full mr-4">
                    <i class="fas fa-bell text-blue-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800">View Notices</h3>
                    <p class="text-sm text-gray-600">Check school announcements</p>
                </div>
            </div>
        </a>

        <a href="events.php" class="bg-white p-6   hover:shadow-md transition-shadow">
            <div class="flex items-center">
                <div class="bg-green-100 p-3 rounded-full mr-4">
                    <i class="fas fa-calendar-alt text-green-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800">School Events</h3>
                    <p class="text-sm text-gray-600">View upcoming events</p>
                </div>
            </div>
        </a>

        <a href="notifications.php" class="bg-white p-6   hover:shadow-md transition-shadow">
            <div class="flex items-center">
                <div class="bg-purple-100 p-3 rounded-full mr-4">
                    <i class="fas fa-bell text-purple-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800">Notifications</h3>
                    <p class="text-sm text-gray-600">View all notifications</p>
                </div>
            </div>
        </a>
    </div>
</div>

<script>
// Enhanced Notification System with Immediate UI Updates
document.addEventListener('DOMContentLoaded', function() {
    console.log('Page loaded - initializing notification system');
    initializeNotificationSystem();
});

function initializeNotificationSystem() {
    // Update button state immediately on page load
    updateNotificationButtonState();
    updateNotificationBadge();
    
    // Set up event listeners
    const notificationButton = document.getElementById('notificationButton');
    const refreshButton = document.getElementById('refreshNotifications');
    
    if (notificationButton) {
        // Remove any existing listeners and add new one
        const newButton = notificationButton.cloneNode(true);
        notificationButton.parentNode.replaceChild(newButton, notificationButton);
        document.getElementById('notificationButton').addEventListener('click', handleNotificationButtonClick);
    }
    
    if (refreshButton) {
        refreshButton.addEventListener('click', manualCheckNotifications);
    }
    
    console.log('Notification system initialized - current permission:', Notification.permission);
}

function handleNotificationButtonClick() {
    const currentPermission = Notification.permission;
    console.log('Button clicked - current permission:', currentPermission);
    
    if (currentPermission === 'default') {
        requestNotificationPermission();
    } else if (currentPermission === 'granted') {
        // If already enabled, show info about how to disable
        if (confirm('Notifications are enabled. To disable them, you need to change the permission in your browser settings. Would you like to test a notification?')) {
            showTestNotification();
        }
    } else if (currentPermission === 'denied') {
        alert('Notifications are blocked. Please enable them in your browser settings to receive notifications.');
    }
}

function updateNotificationButtonState() {
    console.log('Updating button state - permission:', Notification.permission);
    
    if (!('Notification' in window)) {
        setButtonState('unsupported');
        return;
    }
    
    const permission = Notification.permission;
    const button = document.getElementById('notificationButton');
    
    if (!button) return;
    
    switch(permission) {
        case 'granted':
            setButtonState('enabled');
            break;
        case 'denied':
            setButtonState('denied');
            break;
        case 'default':
        default:
            setButtonState('default');
            break;
    }
}

function setButtonState(state) {
    const button = document.getElementById('notificationButton');
    const buttonText = document.getElementById('notificationButtonText');
    
    if (!button || !buttonText) return;
    
    // Clear all classes and set base classes
    button.className = 'px-4 py-2  font-semibold transition-colors flex items-center relative';
    
    switch(state) {
        case 'enabled':
            button.innerHTML = '<i class="fas fa-bell mr-2"></i><span id="notificationButtonText">Notifications Enabled</span>';
            button.classList.add('bg-green-600', 'text-white', 'hover:bg-green-700');
            buttonText.textContent = 'Notifications Enabled';
            console.log('Button state set to: ENABLED (Green)');
            break;
            
        case 'denied':
            button.innerHTML = '<i class="fas fa-bell-slash mr-2"></i><span id="notificationButtonText">Notifications Blocked</span>';
            button.classList.add('bg-red-600', 'text-white', 'hover:bg-red-700');
            buttonText.textContent = 'Notifications Blocked';
            console.log('Button state set to: DENIED (Red)');
            break;
            
        case 'unsupported':
            button.innerHTML = '<i class="fas fa-bell-slash mr-2"></i><span id="notificationButtonText">Not Supported</span>';
            button.classList.add('bg-gray-400', 'text-white', 'cursor-not-allowed');
            buttonText.textContent = 'Not Supported';
            console.log('Button state set to: UNSUPPORTED (Gray)');
            break;
            
        default:
            button.innerHTML = '<i class="fas fa-bell mr-2"></i><span id="notificationButtonText">Enable Notifications</span>';
            button.classList.add('bg-white', 'text-blue-600', 'hover:bg-blue-50', 'border', 'border-blue-300');
            buttonText.textContent = 'Enable Notifications';
            console.log('Button state set to: DEFAULT (Blue)');
            break;
    }
}

function requestNotificationPermission() {
    console.log('Requesting notification permission...');
    
    if (!('Notification' in window)) {
        alert('Your browser does not support notifications.');
        return;
    }
    
    // Show loading state immediately
    const button = document.getElementById('notificationButton');
    const originalContent = button.innerHTML;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i><span>Requesting...</span>';
    button.disabled = true;
    
    Notification.requestPermission().then(function(permission) {
        console.log('Permission result:', permission);
        
        // Re-enable button
        button.disabled = false;
        
        // Update UI immediately
        updateNotificationButtonState();
        
        if (permission === 'granted') {
            console.log('Permission granted - showing welcome notification');
            showBrowserNotification(
                'Notifications Enabled ✅', 
                'You will now receive notifications from Adinoheightschool'
            );
            
            // Test push notification capability
            testPushNotification();
            
        } else if (permission === 'denied') {
            console.log('Permission denied');
            alert('Notifications were blocked. You can enable them later in your browser settings.');
        }
    }).catch(function(error) {
        console.error('Error requesting permission:', error);
        button.disabled = false;
        updateNotificationButtonState();
        alert('Error requesting notification permission: ' + error.message);
    });
}

function showBrowserNotification(title, message) {
    if ('Notification' in window && Notification.permission === 'granted') {
        try {
            const notification = new Notification(title, {
                body: message,
                icon: '/img/logotwo.png',
                badge: '/img/logotwo.png',
                tag: 'welcome-notification'
            });

            notification.onclick = function() {
                console.log('Notification clicked');
                window.focus();
                notification.close();
            };

            // Auto-close after 5 seconds
            setTimeout(() => {
                notification.close();
            }, 5000);
            
            console.log('Browser notification shown:', title);
        } catch (error) {
            console.error('Error showing notification:', error);
        }
    } else {
        console.log('Cannot show notification - permission not granted');
    }
}

function testPushNotification() {
    // Test if we can show another notification after a delay
    setTimeout(() => {
        if (Notification.permission === 'granted') {
            showBrowserNotification(
                'Test Notification', 
                'This is a test notification from Adinoheightschool'
            );
        }
    }, 2000);
}

function showTestNotification() {
    showBrowserNotification(
        'Test Notification', 
        'This is a test notification from Adinoheightschool'
    );
}

function updateNotificationBadge() {
    const badge = document.getElementById('notificationBadge');
    if (badge) {
        const count = <?php echo $unread_notifications; ?>;
        console.log('Updating badge count:', count);
        if (count > 0) {
            badge.textContent = count > 99 ? '99+' : count;
            badge.classList.remove('hidden');
        } else {
            badge.classList.add('hidden');
        }
    }
}

function manualCheckNotifications() {
    console.log('Manual refresh triggered');
    
    const refreshBtn = document.getElementById('refreshNotifications');
    const originalHTML = refreshBtn.innerHTML;
    
    // Show loading state
    refreshBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Checking...';
    refreshBtn.disabled = true;
    
    // Simulate API call
    setTimeout(() => {
        updateNotificationBadge();
        updateNotificationButtonState();
        
        // Show success state
        refreshBtn.innerHTML = '<i class="fas fa-check mr-1"></i> Updated';
        
        setTimeout(() => {
            refreshBtn.innerHTML = originalHTML;
            refreshBtn.disabled = false;
        }, 1500);
        
        console.log('Manual refresh completed');
    }, 1000);
}

// Enhanced error handling
window.addEventListener('error', function(e) {
    console.error('JavaScript error:', e.error);
    console.error('Error details:', e.message, 'at', e.filename, 'line', e.lineno);
});

// Listen for permission changes (though this doesn't work in all browsers)
if ('Notification' in window) {
    // Some browsers support permission change events
    if ('onnotificationpermission' in window) {
        window.onnotificationpermission = function() {
            console.log('Notification permission changed');
            updateNotificationButtonState();
        };
    }
}

console.log('Notification script loaded successfully');
</script>

<?php
include 'include/footer.php';
?>