<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Start session
$sessionName = 'ADINOHEIGHT_SESSION';
session_name($sessionName);

try {
    if (!session_start()) {
        throw new Exception('Failed to start session');
    }
} catch (Exception $e) {
    error_log("Session start failed in download_receipt: " . $e->getMessage());
    header('Location: ../portal_login.php?error=session');
    exit();
}

// Check if user is logged in and is a parent
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true || $_SESSION['user_type'] !== 'parent') {
    // Redirect to correct dashboards based on user type
    if (isset($_SESSION['user_type'])) {
        if ($_SESSION['user_type'] === 'student') {
            header('Location: ../student/index.php');
            exit();
        } elseif ($_SESSION['user_type'] === 'teacher') {
            header('Location: ../teacher/index.php');
            exit();
        }
    }
    // If no valid session, redirect to login
    header('Location: ../portal_login.php');
    exit();
}

// Session timeout (30 minutes)
$inactive = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > $inactive)) {
    session_unset();
    session_destroy();
    header('Location: ../portal_login.php?error=timeout');
    exit();
}
$_SESSION['last_activity'] = time();

// Include database connection
include '../config/conn.php';

// Get parent ID from session
$parent_id = $_SESSION['user_id'];

// Validate payment ID
$payment_id = isset($_GET['payment_id']) ? intval($_GET['payment_id']) : 0;
if ($payment_id <= 0) {
    die("Invalid payment ID");
}

// Get payment details with parent verification
$payment_query = "SELECT p.*, i.description, i.amount as invoice_amount, 
                 s.full_name, s.student_id, s.roll_number, s.parent_id,
                 c.class_name, sec.section_name
                 FROM payments p
                 JOIN invoices i ON p.invoice_id = i.id
                 JOIN students s ON i.student_id = s.id
                 LEFT JOIN classes c ON s.class = c.id
                 LEFT JOIN sections sec ON s.section = sec.id
                 WHERE p.id = ? AND s.parent_id = ?";

$stmt = $conn->prepare($payment_query);
$stmt->bind_param("ii", $payment_id, $parent_id);
$stmt->execute();
$payment_result = $stmt->get_result();

if (!$payment_result || $payment_result->num_rows == 0) {
    // Check if payment exists but doesn't belong to this parent
    $check_payment_query = "SELECT p.id FROM payments p 
                           JOIN invoices i ON p.invoice_id = i.id 
                           JOIN students s ON i.student_id = s.id 
                           WHERE p.id = ?";
    $check_stmt = $conn->prepare($check_payment_query);
    $check_stmt->bind_param("i", $payment_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows > 0) {
        // Payment exists but doesn't belong to this parent
        die("Access denied. You do not have permission to view this receipt.");
    } else {
        // Payment doesn't exist
        die("Payment not found for ID: $payment_id");
    }
}

$payment = $payment_result->fetch_assoc();
$stmt->close();

// Verify the parent has access to this student's data
$verify_parent_query = "SELECT id FROM students WHERE id = ? AND parent_id = ?";
$verify_stmt = $conn->prepare($verify_parent_query);
$student_id_from_payment = $payment['student_id']; // We need to get the actual student ID from payment
// We need to modify the query to get student ID properly

// Let's re-fetch with proper student ID verification
$student_verification_query = "SELECT s.id 
                              FROM students s 
                              JOIN invoices i ON s.id = i.student_id 
                              JOIN payments p ON i.id = p.invoice_id 
                              WHERE p.id = ? AND s.parent_id = ?";
$verify_stmt = $conn->prepare($student_verification_query);
$verify_stmt->bind_param("ii", $payment_id, $parent_id);
$verify_stmt->execute();
$verification_result = $verify_stmt->get_result();

if ($verification_result->num_rows == 0) {
    die("Access denied. You do not have permission to view this receipt.");
}
$verify_stmt->close();

// Include TCPDF library
require_once(__DIR__ . '/../tcpdf/TCPDF-main/tcpdf.php');

// Create new PDF document
$pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);

// Set document info
$pdf->SetCreator('AdinoHeight College');
$pdf->SetAuthor('AdinoHeight College');
$pdf->SetTitle('Payment Receipt - ' . $payment['student_id']);
$pdf->SetSubject('Payment Receipt');

// Add a page
$pdf->AddPage();

// Build HTML content
$html = '
<style>
    .header { text-align: center; margin-bottom: 20px; }
    .school-name { font-size: 20px; font-weight: bold; color: #2c3e50; }
    .receipt-title { font-size: 18px; color: #3498db; margin-top: 5px; }
    .details { margin: 15px 0; }
    .detail-row { margin: 6px 0; }
    .label { font-weight: bold; display: inline-block; width: 120px; }
    .footer { margin-top: 30px; text-align: center; font-size: 12px; color: #7f8c8d; }
    .signature { margin-top: 40px; border-top: 1px solid #000; width: 200px; margin-left: auto; margin-right: auto; }
    table { width: 100%; border-collapse: collapse; margin: 15px 0; }
    th { background-color: #f2f2f2; text-align: left; padding: 8px; border: 1px solid #ddd; }
    td { padding: 8px; border: 1px solid #ddd; }
    .total-row { font-weight: bold; }
</style>

<div class="header">
    <div class="school-name">ADINOHEIGHT COLLEGE</div>
    <div class="receipt-title">PAYMENT RECEIPT</div>
    <div style="font-size: 12px; color: #7f8c8d; margin-top: 5px;">Parent Portal Receipt</div>
</div>

<div class="details">
    <div class="detail-row"><span class="label">Receipt No:</span> REC' . str_pad($payment_id, 6, '0', STR_PAD_LEFT) . '</div>
    <div class="detail-row"><span class="label">Date:</span> ' . date('F j, Y', strtotime($payment['payment_date'])) . '</div>
    <div class="detail-row"><span class="label">Student Name:</span> ' . htmlspecialchars($payment['full_name']) . '</div>
    <div class="detail-row"><span class="label">Student ID:</span> ' . htmlspecialchars($payment['student_id']) . '</div>
    <div class="detail-row"><span class="label">Class:</span> ' . htmlspecialchars($payment['class_name'] . ' - ' . $payment['section_name']) . '</div>
    <div class="detail-row"><span class="label">Roll No:</span> ' . htmlspecialchars($payment['roll_number']) . '</div>
    <div class="detail-row"><span class="label">Parent:</span> ' . htmlspecialchars($_SESSION['full_name']) . '</div>
</div>

<table>
    <tr>
        <th width="70%">Description</th>
        <th width="30%">Amount (₦)</th>
    </tr>
    <tr>
        <td>' . htmlspecialchars($payment['description']) . '</td>
        <td>' . number_format($payment['invoice_amount'], 2) . '</td>
    </tr>
    <tr class="total-row">
        <td style="text-align:right;">Amount Paid:</td>
        <td>' . number_format($payment['amount'], 2) . '</td>
    </tr>
    <tr>
        <td style="text-align:right;">Payment Method:</td>
        <td>' . htmlspecialchars(ucfirst($payment['payment_method'])) . '</td>
    </tr>
    <tr>
        <td style="text-align:right;">Transaction ID:</td>
        <td>' . (!empty($payment['transaction_id']) ? htmlspecialchars($payment['transaction_id']) : 'N/A') . '</td>
    </tr>
    <tr>
        <td style="text-align:right;">Payment Date:</td>
        <td>' . date('M j, Y g:i A', strtotime($payment['payment_date'])) . '</td>
    </tr>
</table>

<div class="footer">
    <div>Thank you for your payment!</div>
    <div>AdinoHeight College<br>www.adinoheightcollege.edu.ng</div>
    <div style="margin-top: 20px; font-size: 10px; color: #95a5a6;">
        Generated on: ' . date('Y-m-d H:i:s') . '<br>
        Generated by: Parent Portal (' . htmlspecialchars($_SESSION['username']) . ')
    </div>
    <div class="signature"></div>
</div>
';

// Output HTML content
$pdf->writeHTML($html, true, false, true, false, '');

// Close database connection
$conn->close();

// Download as PDF
$pdf->Output('receipt_' . $payment['student_id'] . '_' . date('Ymd_His') . '.pdf', 'D');
 ?>