<?php
// Set timezone for Nigeria (UTC+1)
date_default_timezone_set('Africa/Lagos');
ob_start();

include 'include/header.php';
include '../config/conn.php';
require '../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Writer\SvgWriter;

// Check if logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$attendance_date = isset($_GET['attendance_date']) ? $_GET['attendance_date'] : date('Y-m-d');
$class_id = isset($_GET['class_id']) ? intval($_GET['class_id']) : 0;
$section_id = isset($_GET['section_id']) ? intval($_GET['section_id']) : 0;
$success = '';
$error = '';

// Fetch active classes
$classes_query = "SELECT id, class_name FROM classes WHERE status = 'active' ORDER BY class_name";
$classes_result = mysqli_query($conn, $classes_query);

// Fetch sections based on selected class
$sections = [];
if ($class_id > 0) {
    $sections_query = "SELECT id, section_name FROM sections WHERE class_id = $class_id AND status = 'active' ORDER BY section_name";
    $sections_result = mysqli_query($conn, $sections_query);
    if ($sections_result) {
        while ($section = mysqli_fetch_assoc($sections_result)) {
            $sections[] = $section;
        }
    } else {
        $error = "Error fetching sections: " . mysqli_error($conn);
    }
}

// Fetch students based on selected class and section
$students = [];
if ($class_id > 0 && $section_id > 0) {
    $students_query = "SELECT id, student_id, full_name, roll_number, email 
                       FROM students 
                       WHERE class = $class_id AND section = $section_id AND status = 'active' 
                       ORDER BY roll_number";
    $students_result = mysqli_query($conn, $students_query);
    if ($students_result) {
        while ($student = mysqli_fetch_assoc($students_result)) {
            $students[] = $student;
        }
    } else {
        $error = "Error loading students: " . mysqli_error($conn);
    }
}

// Check existing attendance
$existing_attendance = [];
$attendance_exists = false;
if ($class_id > 0 && $section_id > 0 && !empty($attendance_date)) {
    $check_attendance_query = "SELECT student_id, status, recorded_at 
                               FROM attendance 
                               WHERE class_id = $class_id 
                               AND section_id = $section_id 
                               AND attendance_date = '$attendance_date'";
    $check_attendance_result = mysqli_query($conn, $check_attendance_query);
    if ($check_attendance_result) {
        if (mysqli_num_rows($check_attendance_result) > 0) {
            $attendance_exists = true;
            while ($row = mysqli_fetch_assoc($check_attendance_result)) {
                $existing_attendance[$row['student_id']] = $row;
            }
        }
    } else {
        $error = "Error checking attendance: " . mysqli_error($conn);
    }
}

// Calendar variables
$calendar_month = isset($_GET['calendar_month']) ? $_GET['calendar_month'] : date('Y-m');
$month_name = date('F Y', strtotime($calendar_month));
$first_day = date('N', strtotime($calendar_month . '-01'));
$days_in_month = date('t', strtotime($calendar_month . '-01'));

// Get calendar attendance data
$calendar_data = [];
if ($class_id > 0 && $section_id > 0) {
    $calendar_query = "SELECT attendance_date, 
                       SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present,
                       SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent,
                       SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late,
                       SUM(CASE WHEN status = 'half_day' THEN 1 ELSE 0 END) as half_day
                       FROM attendance 
                       WHERE class_id = $class_id 
                       AND section_id = $section_id 
                       AND DATE_FORMAT(attendance_date, '%Y-%m') = '$calendar_month'
                       GROUP BY attendance_date";
    $calendar_result = mysqli_query($conn, $calendar_query);
    if ($calendar_result) {
        while ($row = mysqli_fetch_assoc($calendar_result)) {
            $calendar_data[$row['attendance_date']] = $row;
        }
    } else {
        $error = "Error loading calendar data: " . mysqli_error($conn);
    }
}

// Email notification function for both admin and student
function sendEmailNotification($student_name, $student_email, $attendance_date, $status, $time) {
    global $conn;
    $admin_query = "SELECT email FROM users WHERE role = 'admin' LIMIT 1";
    $admin_result = mysqli_query($conn, $admin_query);
    
    if (mysqli_num_rows($admin_result) === 0) {
        error_log("No admin found in database");
        return false;
    }
    
    $admin = mysqli_fetch_assoc($admin_result);
    $admin_email = $admin['email'];
    
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host       = 'smtp.gmail.com';
        $mail->SMTPAuth   = true;
        $mail->Username   = 'holuwaseunisrael@gmail.com';
        $mail->Password   = 'ceae tepn tfyx oxjn';
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        $mail->Port       = 465;

        $mail->setFrom('attendance@schoolsystem.com', 'Attendance System');
        $mail->addAddress($admin_email);
        if (!empty($student_email)) {
            $mail->addAddress($student_email);
        }
        $mail->addReplyTo('noreply@schoolsystem.com', 'No Reply');

        $mail->isHTML(true);
        $mail->Subject = "Student Attendance - $student_name - $attendance_date";
        $mail->Body = "
        <html>
        <head><title>Attendance Notification</title></head>
        <body>
            <h3>Attendance Recorded</h3>
            <p><strong>Student:</strong> $student_name</p>
            <p><strong>Date:</strong> $attendance_date</p>
            <p><strong>Time:</strong> $time</p>
            <p><strong>Status:</strong> $status</p>
        </body>
        </html>";
        
        if ($mail->send()) {
            error_log("Email sent successfully to admin: $admin_email and student: $student_email");
            return true;
        } else {
            error_log("Email failed to send: " . $mail->ErrorInfo);
            return false;
        }
    } catch (Exception $e) {
        error_log("Mailer Error: {$mail->ErrorInfo}");
        return false;
    }
}

// Handle manual attendance submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_attendance'])) {
    $attendance_date = $_POST['attendance_date'];
    $class_id = intval($_POST['class_id']);
    $section_id = intval($_POST['section_id']);
    $attendance_data = $_POST['attendance'];
    
    if (!empty($attendance_data) && $class_id > 0 && $section_id > 0) {
        mysqli_begin_transaction($conn);
        try {
            // Delete existing attendance for this date, class, and section
            $delete_query = "DELETE FROM attendance WHERE attendance_date = '$attendance_date' AND class_id = $class_id AND section_id = $section_id";
            if (!mysqli_query($conn, $delete_query)) {
                throw new Exception("Error deleting existing attendance: " . mysqli_error($conn));
            }
            
            // Get current time for late calculation
            $current_time = date('H:i:s');
            
            foreach ($attendance_data as $student_id => $status) {
                $student_id = intval($student_id);
                $status = mysqli_real_escape_string($conn, $status);
                
                if ($status === 'present') {
                    // Get student info for email
                    $student_query = "SELECT full_name, email FROM students WHERE id = $student_id";
                    $student_result = mysqli_query($conn, $student_query);
                    
                    if (mysqli_num_rows($student_result) > 0) {
                        $student = mysqli_fetch_assoc($student_result);
                        
                        // Send email notification to admin and student
                        sendEmailNotification(
                            $student['full_name'], 
                            $student['email'], 
                            $attendance_date, 
                            $status, 
                            $current_time
                        );
                    }
                }
                
                $insert_query = "INSERT INTO attendance (student_id, class_id, section_id, attendance_date, status, recorded_by, recorded_at) 
                                VALUES ($student_id, $class_id, $section_id, '$attendance_date', '$status', {$_SESSION['user_id']}, NOW())";
                if (!mysqli_query($conn, $insert_query)) {
                    throw new Exception("Error inserting attendance: " . mysqli_error($conn));
                }
            }
            
            mysqli_commit($conn);
            $success = "Attendance recorded successfully!";
        } catch (Exception $e) {
            mysqli_rollback($conn);
            $error = "Error recording attendance: " . $e->getMessage();
        }
    } else {
        $error = "No attendance data submitted or invalid class/section.";
    }
}

ob_end_flush();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Attendance System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        .attendance-status {
            padding: 8px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 500;
            display: inline-block;
        }
        .status-present {
            background-color: #DCFCE7;
            color: #16A34A;
        }
        .status-absent {
            background-color: #FEE2E2;
            color: #DC2626;
        }
        .status-late {
            background-color: #FEF3C7;
            color: #D97706;
        }
        .status-half_day {
            background-color: #EDE9FE;
            color: #7C3AED;
        }
        .calendar-day {
            height: 80px;
            border: 1px solid #e5e7eb;
            padding: 5px;
            position: relative;
        }
        .calendar-day.today {
            border: 2px solid #3b82f6;
        }
        .calendar-day.has-attendance {
            background-color: #f0f9ff;
        }
        .attendance-stats {
            font-size: 11px;
            margin-top: 5px;
        }
        .qr-section {
            background-color: #f0f9f0;
            border: 1px solid #c6f6d5;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 25px;
        }
        
        .qr-code-container {
            text-align: center;
            margin: 20px 0;
        }
        
        .qr-instructions {
            background-color: #e6fffa;
            border-left: 4px solid #38b2ac;
            padding: 15px;
            margin: 15px 0;
            border-radius: 4px;
        }
        
        .late-warning {
            background-color: #FEF3C7;
            border: 1px solid #F59E0B;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
            font-size: 14px;
        }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        .error-border {
            border-color: #DC2626;
        }
        
        @media print {
            body * {
                visibility: hidden;
            }
            .qr-print-container, .qr-print-container * {
                visibility: visible;
            }
            .qr-print-container {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
            }
            .no-print {
                display: none !important;
            }
        }
    </style>
</head>
<body class="bg-gray-100">
<div class="container mx-auto px-4 py-8">
    <div class="bg-white p-6  ">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold text-gray-800">Student Attendance System</h2>
            <div class="text-sm text-gray-600">
                Current Time: <span id="current-time"><?php echo date('H:i:s'); ?></span>
            </div>
        </div>

        <div id="success-message" class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4 hidden"></div>
        <div id="error-message" class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4 hidden"></div>
        
        <?php if (!empty($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <!-- Late Warning -->
        <?php if (date('H:i') > '08:30'): ?>
            <div class="late-warning mb-4">
                <i class="fas fa-exclamation-triangle mr-2"></i>
                <strong>Note:</strong> It's after 8:30 AM. Students marking attendance now will be marked as <span class="status-late">Late</span>.
            </div>
        <?php endif; ?>

        <!-- QR Code Attendance Section -->
        <div class="qr-section">
            <h3 class="text-lg font-semibold text-green-800 mb-3">Yearly QR Code Attendance</h3>
            
            <div class="qr-instructions">
                <h4 class="font-semibold mb-2">How to use:</h4>
                <ol class="list-decimal pl-5 space-y-1">
                    <li>Select class and section</li>
                    <li>Generate a yearly QR code for attendance</li>
                    <li>Students scan the QR code with any QR code reader app on their phones</li>
                    <li>They will be directed to a form to enter their Student ID and select date</li>
                    <li>Attendance is automatically recorded upon submission</li>
                    <li><strong>After 8:30 AM, students will be marked as Late</strong></li>
                    <li>Email notifications will be sent to both admin and student</li>
                    <li><strong>This QR code will be valid for 1 year</strong></li>
                </ol>
            </div>
            
            <form id="qr-form" class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Select Class</label>
                    <select name="qr_class_id" id="qr_class_id" class="w-full px-4 py-2 border rounded focus:outline-none focus:ring-2 focus:ring-green-500" required>
                        <option value="">Select Class</option>
                        <?php 
                        mysqli_data_seek($classes_result, 0);
                        while ($class = mysqli_fetch_assoc($classes_result)): 
                        ?>
                            <option value="<?php echo $class['id']; ?>" <?php echo ($class_id == $class['id']) ? 'selected' : ''; ?>>
                                <?php echo $class['class_name']; ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Select Section</label>
                    <select name="qr_section_id" id="qr_section_id" class="w-full px-4 py-2 border rounded focus:outline-none focus:ring-2 focus:ring-green-500" required>
                        <option value="">Select Section</option>
                        <?php foreach ($sections as $section): ?>
                            <option value="<?php echo $section['id']; ?>" <?php echo ($section_id == $section['id']) ? 'selected' : ''; ?>>
                                <?php echo $section['section_name']; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="flex items-end">
                    <button type="submit" id="generate-qr-btn" 
                        class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded flex items-center h-10">
                        <i class="fas fa-qrcode mr-2"></i> Generate Yearly QR Code
                    </button>
                </div>
            </form>
            
            <div id="qr-result-container" class="hidden">
                <div class="qr-code-container">
                    <h4 class="font-semibold text-gray-700 mb-2">Scan this QR code with any mobile app:</h4>
                    <div id="qr-image-container"></div>
                    <p id="qr-expiry-time" class="text-sm text-green-600 font-semibold mt-2"></p>
                    
                    <div class="mt-4 p-3 bg-blue-50 ">
                      <p class="text-sm text-blue-800">
                        <strong>Alternative URL for manual entry:</strong><br>
                        <span id="qr-alternative-url"></span>
                      </p>
                    </div>
                    
                    <!-- Print button added here -->
                    <div class="mt-4">
                        <button onclick="printQRCode()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded flex items-center mx-auto">
                            <i class="fas fa-print mr-2"></i> Print QR Code
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filter Section -->
        <div class="mb-6 bg-gray-50 p-4 ">
            <form id="filter-form" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <!-- Class Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Select Class</label>
                    <select name="class_id" id="class_filter" class="w-full px-4 py-2 border rounded focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        <option value="">Select Class</option>
                        <?php 
                        mysqli_data_seek($classes_result, 0);
                        while ($class = mysqli_fetch_assoc($classes_result)): 
                        ?>
                            <option value="<?php echo $class['id']; ?>" <?php echo ($class_id == $class['id']) ? 'selected' : ''; ?>>
                                <?php echo $class['class_name']; ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <!-- Section Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Select Section</label>
                    <select name="section_id" id="section_filter" class="w-full px-4 py-2 border rounded focus:outline-none focus:ring-2 focus:ring-blue-500" <?php echo ($class_id == 0) ? 'disabled' : ''; ?> required>
                        <option value="">Select Section</option>
                        <?php foreach ($sections as $section): ?>
                            <option value="<?php echo $section['id']; ?>" <?php echo ($section_id == $section['id']) ? 'selected' : ''; ?>>
                                <?php echo $section['section_name']; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Date Selection -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Select Date</label>
                    <input type="date" name="attendance_date" id="attendance_date" value="<?php echo $attendance_date; ?>" 
                        class="w-full px-4 py-2 border rounded focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                </div>
                
                <!-- Buttons -->
                <div class="flex items-end space-x-2">
                    <button type="submit" id="load-students-btn" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded flex items-center h-10">
                        <i class="fas fa-filter mr-2"></i> Load Students
                    </button>
                    <button type="button" id="clear-btn" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded flex items-center h-10">
                        <i class="fas fa-times mr-2"></i> Clear
                    </button>
                </div>
            </form>
            
            <?php if (!empty($class_id) && !empty($section_id)): ?>
                <div class="mt-4">
                    <a href="export_attendance_month.php?class_id=<?php echo (int)$class_id; ?>&section_id=<?php echo (int)$section_id; ?>&month=<?php echo htmlspecialchars($calendar_month); ?>"
                       class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded flex items-center inline-block">
                       <i class="fas fa-file-excel mr-2"></i> Export Monthly Attendance
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <div id="attendance-container">
            <?php if ($class_id > 0 && $section_id > 0): ?>
                <!-- Attendance Form -->
                <div class="mb-6 bg-white p-4  ">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-lg font-semibold text-gray-800">
                            Mark Student Attendance for <?php 
                            $class_name = mysqli_fetch_assoc(mysqli_query($conn, "SELECT class_name FROM classes WHERE id = $class_id"))['class_name'];
                            $section_name = mysqli_fetch_assoc(mysqli_query($conn, "SELECT section_name FROM sections WHERE id = $section_id"))['section_name'];
                            echo $class_name . " - " . $section_name . " (" . date('d M Y', strtotime($attendance_date)) . ")";
                            ?>
                        </h3>
                        
                        <?php if ($attendance_exists): ?>
                            <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 rounded-full">
                                <i class="fas fa-info-circle mr-1"></i> Attendance already recorded for this date
                            </span>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (!empty($students)): ?>
                        <form id="attendance-form" method="POST" action="">
                            <input type="hidden" name="class_id" value="<?php echo $class_id; ?>">
                            <input type="hidden" name="section_id" value="<?php echo $section_id; ?>">
                            <input type="hidden" name="attendance_date" value="<?php echo $attendance_date; ?>">
                            <input type="hidden" name="submit_attendance" value="1">
                            
                            <div class="overflow-x-auto">
                                <table class="min-w-full bg-white">
                                    <thead class="bg-gray-200 text-gray-700">
                                        <tr>
                                            <th class="py-3 px-4 text-left">Roll No</th>
                                            <th class="py-3 px-4 text-left">Student ID</th>
                                            <th class="py-3 px-4 text-left">Name</th>
                                            <th class="py-3 px-4 text-left">Status</th>
                                            <th class="py-3 px-4 text-left">Time</th>
                                        </tr>
                                    </thead>
                                    <tbody class="text-gray-600">
                                        <?php foreach ($students as $student): 
                                            $status = isset($existing_attendance[$student['id']]) ? $existing_attendance[$student['id']]['status'] : '';
                                            $time = isset($existing_attendance[$student['id']]) ? date('H:i', strtotime($existing_attendance[$student['id']]['recorded_at'])) : '';
                                        ?>
                                            <tr class="border-b border-gray-200 hover:bg-gray-50">
                                                <td class="py-3 px-4"><?php echo $student['roll_number']; ?></td>
                                                <td class="py-3 px-4"><?php echo $student['student_id']; ?></td>
                                                <td class="py-3 px-4"><?php echo $student['full_name']; ?></td>
                                                <td class="py-3 px-4">
                                                    <div class="flex flex-wrap gap-4">
                                                        <label class="inline-flex items-center">
                                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="present" 
                                                                <?php echo ($status == 'present') ? 'checked' : 'checked'; ?> 
                                                                class="form-radio h-4 w-4 text-green-600">
                                                            <span class="ml-2 text-green-600">Present</span>
                                                        </label>
                                                        <label class="inline-flex items-center">
                                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="absent" 
                                                                <?php echo ($status == 'absent') ? 'checked' : ''; ?> 
                                                                class="form-radio h-4 w-4 text-red-600">
                                                            <span class="ml-2 text-red-600">Absent</span>
                                                        </label>
                                                        <label class="inline-flex items-center">
                                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="late" 
                                                                <?php echo ($status == 'late') ? 'checked' : ''; ?> 
                                                                class="form-radio h-4 w-4 text-yellow-600">
                                                            <span class="ml-2 text-yellow-600">Late</span>
                                                        </label>
                                                        <label class="inline-flex items-center">
                                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="half_day" 
                                                                <?php echo ($status == 'half_day') ? 'checked' : ''; ?> 
                                                                class="form-radio h-4 w-4 text-purple-600">
                                                            <span class="ml-2 text-purple-600">Half Day</span>
                                                        </label>
                                                    </div>
                                                </td>
                                                <td class="py-3 px-4">
                                                    <?php if (!empty($time)): ?>
                                                        <span class="attendance-status status-<?php echo $status; ?>">
                                                            <?php echo $time; ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <div class="mt-6 flex justify-center space-x-4">
                                <button type="button" onclick="selectAllPresent()" class="bg-green-500 hover:bg-green-600 text-white px-4 py-2 rounded">
                                    <i class="fas fa-check-circle mr-2"></i> Mark All Present
                                </button>
                                <button type="button" onclick="selectAllAbsent()" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded">
                                    <i class="fas fa-times-circle mr-2"></i> Mark All Absent
                                </button>
                                <button type="submit" id="save-attendance-btn" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded flex items-center">
                                    <i class="fas fa-save mr-2"></i> Save Attendance
                                </button>
                            </div>
                        </form>
                    <?php else: ?>
                        <p class="text-center py-4 text-gray-500">
                            No active students found in this class and section.
                        </p>
                    <?php endif; ?>
                </div>
                
                <!-- Attendance Calendar View -->
                <div class="mt-6 bg-white p-4  ">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Attendance Calendar</h3>
                    
                    <div class="mb-4 flex justify-between items-center">
                        <h4 class="text-md font-medium text-gray-700"><?php echo $month_name; ?></h4>
                        
                        <form method="GET" action="" class="flex items-center">
                            <input type="hidden" name="class_id" value="<?php echo $class_id; ?>">
                            <input type="hidden" name="section_id" value="<?php echo $section_id; ?>">
                            <input type="hidden" name="attendance_date" value="<?php echo $attendance_date; ?>">
                            <input type="month" name="calendar_month" value="<?php echo $calendar_month; ?>" 
                                class="px-3 py-1 border rounded focus:outline-none focus:ring-2 focus:ring-blue-500 mr-2">
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 ">
                                View
                            </button>
                        </form>
                    </div>
                    
                    <div class="grid grid-cols-7 gap-2 mb-2">
                        <div class="text-center font-medium text-gray-700">Mon</div>
                        <div class="text-center font-medium text-gray-700">Tue</div>
                        <div class="text-center font-medium text-gray-700">Wed</div>
                        <div class="text-center font-medium text-gray-700">Thu</div>
                        <div class="text-center font-medium text-gray-700">Fri</div>
                        <div class="text-center font-medium text-gray-700">Sat</div>
                        <div class="text-center font-medium text-gray-700">Sun</div>
                    </div>
                    
                    <div class="grid grid-cols-7 gap-2">
                        <?php
                        // Add empty cells for days before the first day of the month
                        for ($i = 1; $i < $first_day; $i++) {
                            echo '<div class="calendar-day bg-gray-100 rounded"></div>';
                        }
                        
                        // Add cells for each day of the month
                        for ($day = 1; $day <= $days_in_month; $day++) {
                            $current_date = date('Y-m-d', strtotime($calendar_month . '-' . sprintf('%02d', $day)));
                            $is_today = ($current_date == date('Y-m-d')) ? 'today' : '';
                            $has_attendance = isset($calendar_data[$current_date]);
                            
                            echo '<div class="calendar-day rounded ' . $is_today . ' ' . ($has_attendance ? 'has-attendance' : '') . '">';
                            echo '<div class="text-sm font-medium">' . $day . '</div>';
                            
                            if ($has_attendance) {
                                $data = $calendar_data[$current_date];
                                echo '<div class="attendance-stats">';
                                echo '<div class="text-green-600">P: ' . $data['present'] . '</div>';
                                echo '<div class="text-red-600">A: ' . $data['absent'] . '</div>';
                                echo '<div class="text-yellow-600">L: ' . $data['late'] . '</div>';
                                echo '<div class="text-purple-600">H: ' . $data['half_day'] . '</div>';
                                echo '</div>';
                            }
                            
                            echo '</div>';
                            
                            // Start new row after Sunday
                            if (($first_day + $day - 1) % 7 == 0 && $day != $days_in_month) {
                                echo '</div><div class="grid grid-cols-7 gap-2">';
                            }
                        }
                        
                        // Add empty cells for remaining days in the last week
                        $last_day = date('N', strtotime($calendar_month . '-' . $days_in_month));
                        if ($last_day < 7) {
                            for ($i = $last_day; $i < 7; $i++) {
                                echo '<div class="calendar-day bg-gray-100 rounded"></div>';
                            }
                        }
                        ?>
                    </div>
                    
                    <div class="mt-4 flex justify-center items-center space-x-4 text-xs">
                        <div class="flex items-center">
                            <div class="w-4 h-4 bg-blue-100 mr-1 border border-blue-300 rounded"></div>
                            <span>Today</span>
                        </div>
                        <div class="flex items-center">
                            <div class="w-4 h-4 bg-blue-50 mr-1 rounded"></div>
                            <span>Has Attendance</span>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div class="bg-white p-8  text-center">
                    <i class="fas fa-calendar-alt text-4xl text-gray-300 mb-4"></i>
                    <h3 class="text-lg font-medium text-gray-600">Please select a class and section to view attendance</h3>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
// Utility functions for manual attendance
function selectAllPresent() {
    const radios = document.querySelectorAll('input[type="radio"][value="present"]');
    radios.forEach(radio => {
        radio.checked = true;
    });
}

function selectAllAbsent() {
    const radios = document.querySelectorAll('input[type="radio"][value="absent"]');
    radios.forEach(radio => {
        radio.checked = true;
    });
}

// Print QR Code function
function printQRCode() {
    const qrImage = document.querySelector('#qr-image-container img');
    const expiryTime = document.querySelector('#qr-expiry-time').textContent;
    const alternativeUrl = document.querySelector('#qr-alternative-url').textContent;
    
    if (!qrImage) {
        alert('No QR code to print. Please generate a QR code first.');
        return;
    }
    
    // Create a print-friendly version
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`
        <!DOCTYPE html>
        <html>
        <head>
            <title>Print QR Code</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    text-align: center;
                    padding: 20px;
                }
                .qr-container {
                    margin: 20px auto;
                    max-width: 500px;
                }
                .qr-image {
                    max-width: 100%;
                    height: auto;
                    border: 1px solid #ddd;
                    padding: 10px;
                    background: white;
                }
                .details {
                    margin-top: 20px;
                    text-align: left;
                    background: #f9f9f9;
                    padding: 15px;
                    border-radius: 5px;
                }
                .instruction {
                    margin-top: 15px;
                    font-size: 14px;
                    color: #555;
                }
                @media print {
                    body {
                        padding: 0;
                    }
                    .no-print {
                        display: none;
                    }
                }
            </style>
        </head>
        <body>
            <h2>Yearly Attendance QR Code</h2>
            <div class="qr-container">
                <img class="qr-image" src="${qrImage.src}" alt="Attendance QR Code">
                <p><strong>${expiryTime}</strong></p>
                
                <div class="details">
                    <p><strong>Alternative URL:</strong><br>${alternativeUrl}</p>
                    <p><strong>Instructions:</strong> Scan this QR code with any QR code reader app to mark attendance.</p>
                </div>
                
                <div class="instruction">
                    <p>Generated on: ${new Date().toLocaleDateString()}</p>
                </div>
            </div>
            
            <div class="no-print" style="margin-top: 20px;">
                <button onclick="window.print()" style="padding: 10px 20px; background: #3b82f6; color: white; border: none; border-radius: 5px; cursor: pointer;">
                    Print
                </button>
                <button onclick="window.close()" style="padding: 10px 20px; background: #6b7280; color: white; border: none; border-radius: 5px; cursor: pointer; margin-left: 10px;">
                    Close
                </button>
            </div>
            
            <script>
                window.onload = function() {
                    // Auto-print when the window loads
                    window.print();
                };
            <\/script>
        </body>
        </html>
    `);
    printWindow.document.close();
}

// Update current time every second
function updateCurrentTime() {
    const now = new Date();
    const timeString = now.toLocaleTimeString();
    document.getElementById('current-time').textContent = timeString;
}

setInterval(updateCurrentTime, 1000);
updateCurrentTime(); // Initial call

// Dynamic section loading based on class selection
document.getElementById('class_filter').addEventListener('change', function() {
    const classId = this.value;
    const sectionSelect = document.getElementById('section_filter');
    
    if (classId) {
        // Enable section dropdown
        sectionSelect.disabled = false;
        
        // Fetch sections for the selected class via AJAX
        fetch('get_sectionss.php?class_id=' + classId)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(sections => {
                // Clear existing options except the first one
                while (sectionSelect.options.length > 1) {
                    sectionSelect.remove(1);
                }
                
                // Add new options
                sections.forEach(section => {
                    const option = document.createElement('option');
                    option.value = section.id;
                    option.textContent = section.section_name;
                    sectionSelect.appendChild(option);
                });
            })
            .catch(error => {
                console.error('Error fetching sections:', error);
                document.getElementById('error-message').textContent = 'Error loading sections: ' + error.message;
                document.getElementById('error-message').classList.remove('hidden');
            });
    } else {
        // Disable and clear section dropdown if no class is selected
        sectionSelect.disabled = true;
        while (sectionSelect.options.length > 1) {
            sectionSelect.remove(1);
        }
    }
});

// Update QR form when class/section filters change
document.getElementById('class_filter').addEventListener('change', function() {
    document.getElementById('qr_class_id').value = this.value;
    
    // Also update the section dropdown for QR form
    const classId = this.value;
    const qrSectionSelect = document.getElementById('qr_section_id');
    
    if (classId) {
        fetch('get_sectionss.php?class_id=' + classId)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(sections => {
                // Clear existing options except the first one
                while (qrSectionSelect.options.length > 1) {
                    qrSectionSelect.remove(1);
                }
                
                // Add new options
                sections.forEach(section => {
                    const option = document.createElement('option');
                    option.value = section.id;
                    option.textContent = section.section_name;
                    qrSectionSelect.appendChild(option);
                });
            })
            .catch(error => {
                console.error('Error fetching sections:', error);
                document.getElementById('error-message').textContent = 'Error loading sections: ' + error.message;
                document.getElementById('error-message').classList.remove('hidden');
            });
    } else {
        // Clear section dropdown if no class is selected
        while (qrSectionSelect.options.length > 1) {
            qrSectionSelect.remove(1);
        }
    }
});

document.getElementById('section_filter').addEventListener('change', function() {
    document.getElementById('qr_section_id').value = this.value;
});

// AJAX for QR code generation
$(document).ready(function() {
    $('#qr-form').on('submit', function(e) {
        e.preventDefault();
        
        // Show loading state
        const btn = $('#generate-qr-btn');
        const originalText = btn.html();
        btn.html('<span class="loading mr-2"></span> Generating...').prop('disabled', true);
        
        // Hide any previous messages
        $('#success-message, #error-message').addClass('hidden');
        
        // Submit form via AJAX
        $.ajax({
            url: 'ajax_generate_qr.php',
            type: 'POST',
            data: $(this).serialize(),
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    // Show QR code
                    $('#qr-image-container').html('<img src="' + response.qr_filename + '" alt="Attendance QR Code" class="mx-auto border-4 border-white shadow-lg" style="max-width: 300px;">');
                    $('#qr-expiry-time').text('This QR code will expire on ' + new Date(response.expiry_time).toLocaleDateString());
                    $('#qr-alternative-url').text(response.alternative_url);
                    $('#qr-result-container').removeClass('hidden');
                    
                    // Show success message
                    $('#success-message').text(response.message).removeClass('hidden');
                } else {
                    // Show error message
                    $('#error-message').text(response.message).removeClass('hidden');
                }
            },
            error: function(xhr, status, error) {
                $('#error-message').text('An error occurred while generating the QR code: ' + error).removeClass('hidden');
            },
            complete: function() {
                // Restore button
                btn.html(originalText).prop('disabled', false);
            }
        });
    });
    
    // AJAX for attendance submission
    $('#attendance-form').on('submit', function(e) {
        e.preventDefault();
        
        // Show loading state
        const btn = $('#save-attendance-btn');
        const originalText = btn.html();
        btn.html('<span class="loading mr-2"></span> Saving...').prop('disabled', true);
        
        // Hide any previous messages
        $('#success-message, #error-message').addClass('hidden');
        
        // Submit form via AJAX
        $.ajax({
            url: 'ajax_save_attendance.php',
            type: 'POST',
            data: $(this).serialize(),
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    // Show success message
                    $('#success-message').text(response.message).removeClass('hidden');
                    
                    // Update any existing attendance times if needed
                    if (response.updated_records) {
                        // You could update the UI here if needed
                        location.reload(); // Or update specific elements instead of reloading
                    }
                } else {
                    // Show error message
                    $('#error-message').text(response.message).removeClass('hidden');
                }
            },
            error: function(xhr, status, error) {
                $('#error-message').text('An error occurred while saving attendance: ' + error).removeClass('hidden');
            },
            complete: function() {
                // Restore button
                btn.html(originalText).prop('disabled', false);
            }
        });
    });
    
    // AJAX for loading students without page reload
    $('#filter-form').on('submit', function(e) {
        e.preventDefault();
        
        const classId = $('#class_filter').val();
        const sectionId = $('#section_filter').val();
        const attendanceDate = $('#attendance_date').val();
        
        if (!classId || !sectionId || !attendanceDate) {
            $('#error-message').text('Please select class, section, and date.').removeClass('hidden');
            return;
        }
        
        // Show loading state
        const btn = $('#load-students-btn');
        const originalText = btn.html();
        btn.html('<span class="loading mr-2"></span> Loading...').prop('disabled', true);
        
        // Hide any previous messages
        $('#success-message, #error-message').addClass('hidden');
        
        // Submit form via AJAX
        $.ajax({
            url: 'ajax_load_students.php',
            type: 'POST',
            data: $(this).serialize(),
            success: function(response) {
                $('#attendance-container').html(response);
            },
            error: function(xhr, status, error) {
                $('#error-message').text('An error occurred while loading students: ' + error).removeClass('hidden');
                $('#attendance-container').html('<div class="bg-white p-8  text-center"><i class="fas fa-exclamation-triangle text-4xl text-red-300 mb-4"></i><h3 class="text-lg font-medium text-red-600">Error loading students. Please try again.</h3></div>');
            },
            complete: function() {
                // Restore button
                btn.html(originalText).prop('disabled', false);
            }
        });
    });
    
    // Clear button functionality
    $('#clear-btn').on('click', function() {
        $('#class_filter').val('');
        $('#section_filter').val('').prop('disabled', true);
        $('#attendance_date').val('<?php echo date('Y-m-d'); ?>');
        $('#attendance-container').html('<div class="bg-white p-8  text-center"><i class="fas fa-calendar-alt text-4xl text-gray-300 mb-4"></i><h3 class="text-lg font-medium text-gray-600">Please select a class and section to view attendance</h3></div>');
    });
});
</script>

<?php
include 'include/footer.php';
mysqli_close($conn);
?>