<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Handle promotion action
if (isset($_POST['promote_students'])) {
    $current_class = mysqli_real_escape_string($conn, $_POST['current_class']);
    $current_section = mysqli_real_escape_string($conn, $_POST['current_section']);
    $target_class = mysqli_real_escape_string($conn, $_POST['target_class']);
    $target_section = mysqli_real_escape_string($conn, $_POST['target_section']);
    
    if (isset($_POST['student_ids']) && is_array($_POST['student_ids']) && !empty($_POST['student_ids'])) {
        $student_ids = array_map(function($id) use ($conn) {
            return mysqli_real_escape_string($conn, $id);
        }, $_POST['student_ids']);
        
        $ids_string = "'" . implode("','", $student_ids) . "'";
        
        // Update students with new class and section
        $update_query = "UPDATE students SET class = '$target_class', section = '$target_section' 
                         WHERE id IN ($ids_string) AND class = '$current_class' AND section = '$current_section'";
        
        if (mysqli_query($conn, $update_query)) {
            $affected_rows = mysqli_affected_rows($conn);
            $success_message = "Successfully promoted $affected_rows student(s) to new class!";
        } else {
            $error_message = "Error promoting students: " . mysqli_error($conn);
        }
    } else {
        $error_message = "Please select at least one student to promote.";
    }
}

// Fetch all active classes
$classes_query = "SELECT id, class_name FROM classes WHERE status = 'active' ORDER BY class_name";
$classes_result = mysqli_query($conn, $classes_query);

// Get current filter values from request
$current_class_filter = isset($_GET['class_filter']) ? mysqli_real_escape_string($conn, $_GET['class_filter']) : '';
$current_section_filter = isset($_GET['section_filter']) ? mysqli_real_escape_string($conn, $_GET['section_filter']) : '';

ob_end_flush();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Promotion System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        .student-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .student-table th, .student-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e5e7eb;
        }
        
        .student-table th {
            background-color: #f9fafb;
            font-weight: 600;
        }
        
        .student-table tr:hover {
            background-color: #f3f4f6;
        }
        
        .promotion-form {
            background-color: #f0f9ff;
            border: 1px solid #bae6fd;
            border-radius: 8px;
            padding: 20px;
            margin-top: 20px;
        }
    </style>
</head>
<body class="bg-gray-100">
<div class="container mx-auto px-4 py-8">
    <div class="bg-white p-6  ">
        <div class="mb-6">
            <h2 class="text-2xl font-bold text-gray-800">Student Promotion</h2>
            <p class="text-gray-600">Promote students from one class to another</p>
        </div>

        <div id="success-message" class="bg-green-100 border border-green-400 text-green-700 px-4 py-3  mb-4 hidden"></div>
        <div id="error-message" class="bg-red-100 border border-red-400 text-red-700 px-4 py-3  mb-4 hidden"></div>
        
        <?php if (isset($success_message)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3  mb-4">
                <?php echo $success_message; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($error_message)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3  mb-4">
                <?php echo $error_message; ?>
            </div>
        <?php endif; ?>

        <!-- Filter Section -->
        <div class="mb-6 bg-gray-50 p-4 ">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4" id="filter-form">
                <!-- Current Class Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Current Class</label>
                    <select name="class_filter" id="class_filter" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        <option value="">Select Class</option>
                        <?php 
                        mysqli_data_seek($classes_result, 0);
                        while ($class = mysqli_fetch_assoc($classes_result)): ?>
                            <option value="<?php echo $class['id']; ?>" <?php echo ($current_class_filter == $class['id']) ? 'selected' : ''; ?>>
                                <?php echo $class['class_name']; ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <!-- Current Section Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Current Section</label>
                    <select name="section_filter" id="section_filter" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" <?php echo empty($current_class_filter) ? 'disabled' : ''; ?> required>
                        <option value="">Select Section</option>
                        <!-- Sections will be loaded via AJAX -->
                    </select>
                </div>
                
                <!-- Filter Button -->
                <div class="flex items-end">
                    <button type="button" id="load-students-btn" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2  flex items-center h-10">
                        <i class="fas fa-filter mr-2"></i> Load Students
                    </button>
                </div>
            </div>
        </div>

        <div id="promotion-container">
            <!-- Content will be loaded via AJAX -->
            <?php if (!empty($current_class_filter) && !empty($current_section_filter)): ?>
                <div class="text-center py-4 text-gray-500">
                    Click "Load Students" to view students for promotion
                </div>
            <?php else: ?>
                <div class="text-center py-8 text-gray-500">
                    <i class="fas fa-users text-4xl text-gray-300 mb-4"></i>
                    <p>Select a class and section to view students</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Initialize page
    const currentClassId = '<?php echo $current_class_filter; ?>';
    const currentSectionId = '<?php echo $current_section_filter; ?>';
    
    // If we have a class filter from page load, populate sections
    if (currentClassId) {
        loadSections(currentClassId, 'section_filter', currentSectionId);
    }
    
    // If we have both filters from page load, load students
    if (currentClassId && currentSectionId) {
        setTimeout(() => {
            loadStudents(currentClassId, currentSectionId);
        }, 500);
    }
    
    // Load sections when class changes
    $('#class_filter').on('change', function() {
        const classId = $(this).val();
        const sectionSelect = $('#section_filter');
        
        if (classId) {
            sectionSelect.prop('disabled', false);
            loadSections(classId, 'section_filter');
        } else {
            sectionSelect.prop('disabled', true);
            sectionSelect.html('<option value="">Select Section</option>');
            // Clear students container
            $('#promotion-container').html(
                '<div class="text-center py-4 text-gray-500">Select a class and section to view students</div>'
            );
        }
    });
    
    // Load students when button is clicked
    $('#load-students-btn').on('click', function() {
        const classId = $('#class_filter').val();
        const sectionId = $('#section_filter').val();
        
        if (!classId || !sectionId) {
            showError('Please select both class and section');
            return;
        }
        
        loadStudents(classId, sectionId);
    });
    
    // Handle promotion form submission
    $(document).on('submit', '#promotion-form', function(e) {
        e.preventDefault();
        
        // Validate form
        const selectedCount = $('.student-checkbox:checked').length;
        const targetClass = $('#target_class').val();
        const targetSection = $('#target_section').val();
        
        if (selectedCount === 0) {
            showError('Please select at least one student to promote.');
            return false;
        }
        
        if (!targetClass || !targetSection) {
            showError('Please select both target class and section.');
            return false;
        }
        
        // Show confirmation dialog
        const targetClassText = $('#target_class option:selected').text();
        const targetSectionText = $('#target_section option:selected').text();
        
        if (!confirm(`Are you sure you want to promote ${selectedCount} student(s) to ${targetClassText} - ${targetSectionText}?`)) {
            return false;
        }
        
        // Submit form via AJAX
        const formData = $(this).serialize();
        
        $.ajax({
            url: 'ajax_promote_students.php',
            type: 'POST',
            data: formData,
            dataType: 'json',
            beforeSend: function() {
                $('#promote-btn').html('<span class="loading mr-2"></span> Promoting...').prop('disabled', true);
            },
            success: function(response) {
                if (response.success) {
                    showSuccess(response.message);
                    // Reload the students list to reflect changes
                    loadStudents($('#class_filter').val(), $('#section_filter').val());
                } else {
                    showError(response.message);
                }
            },
            error: function() {
                showError('An error occurred while promoting students.');
            },
            complete: function() {
                $('#promote-btn').html('<i class="fas fa-graduation-cap mr-2"></i> Promote Selected Students').prop('disabled', false);
            }
        });
    });
    
    // Target class change event
    $(document).on('change', '#target_class', function() {
        const classId = $(this).val();
        const targetSectionSelect = $('#target_section');
        
        if (classId) {
            loadSections(classId, 'target_section');
            targetSectionSelect.prop('disabled', false);
        } else {
            targetSectionSelect.prop('disabled', true);
            targetSectionSelect.html('<option value="">Select New Section</option>');
        }
    });
    
    // Select all functionality
    $(document).on('change', '#select_all', function() {
        $('.student-checkbox').prop('checked', this.checked);
    });
    
    // Individual checkbox change to update "Select All" status
    $(document).on('change', '.student-checkbox', function() {
        const allChecked = $('.student-checkbox:checked').length === $('.student-checkbox').length;
        $('#select_all').prop('checked', allChecked);
    });
});

// Function to load sections via AJAX
function loadSections(classId, targetSelectId, selectedSectionId = '') {
    $.ajax({
        url: 'get_section.php',
        type: 'GET',
        data: { class_id: classId },
        dataType: 'json',
        success: function(sections) {
            const sectionSelect = $('#' + targetSelectId);
            // Clear existing options except the first one
            sectionSelect.html('<option value="">Select Section</option>');
            
            // Add new options
            sections.forEach(section => {
                const option = $('<option>', {
                    value: section.id,
                    text: section.section_name
                });
                
                if (selectedSectionId && section.id == selectedSectionId) {
                    option.prop('selected', true);
                }
                
                sectionSelect.append(option);
            });
        },
        error: function() {
            showError('Error loading sections');
        }
    });
}

// Function to load students via AJAX
function loadStudents(classId, sectionId) {
    // Show loading indicator
    $('#promotion-container').html(
        '<div class="text-center py-8"><i class="fas fa-spinner fa-spin text-blue-500 text-2xl"></i><p class="mt-2 text-gray-600">Loading students...</p></div>'
    );
    
    $.ajax({
        url: 'get_students.php',
        type: 'GET',
        data: { 
            class_id: classId, 
            section_id: sectionId,
            for_promotion: true
        },
        success: function(response) {
            $('#promotion-container').html(response);
        },
        error: function() {
            $('#promotion-container').html(
                '<div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 ">Error loading students. Please try again.</div>'
            );
        }
    });
}

// Utility functions for messages
function showSuccess(message) {
    $('#success-message').text(message).removeClass('hidden');
    $('#error-message').addClass('hidden');
    
    // Auto hide after 5 seconds
    setTimeout(() => {
        $('#success-message').addClass('hidden');
    }, 5000);
}

function showError(message) {
    $('#error-message').text(message).removeClass('hidden');
    $('#success-message').addClass('hidden');
    
    // Auto hide after 5 seconds
    setTimeout(() => {
        $('#error-message').addClass('hidden');
    }, 5000);
}
</script>

<?php
include 'include/footer.php';
mysqli_close($conn);
?>