<?php
// exportTest.php

ob_start();
session_start();

// Secure session cookie
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
    'httponly' => true,
    'samesite' => 'Lax'
]);

// Check login & role
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header("Location: ../login.php?access=denied");
    exit();
}

// Session expiry check
$inactive = 3600;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > $inactive)) {
    session_unset();
    session_destroy();
    header("Location: ../login.php?timeout=1");
    exit();
}
$_SESSION['last_activity'] = time();

// DB connection
include '../config/conn.php';

// Validate quiz_id
if (!isset($_GET['quiz_id']) || !is_numeric($_GET['quiz_id'])) {
    die("Invalid request: quiz_id is required.");
}
$quiz_id = intval($_GET['quiz_id']);

// Fetch test details
$test_query = "SELECT q.*, s.subject_name 
               FROM quizzes q 
               JOIN subjects s ON q.subject_id = s.id 
               WHERE q.id = ?";
$stmt = mysqli_prepare($conn, $test_query);
mysqli_stmt_bind_param($stmt, "i", $quiz_id);
mysqli_stmt_execute($stmt);
$test_result = mysqli_stmt_get_result($stmt);
$selected_test = mysqli_fetch_assoc($test_result);

if (!$selected_test) {
    die("Test not found.");
}

// Fetch results
$results_query = "SELECT qr.*, st.full_name, st.student_id, c.class_name, sec.section_name
                  FROM quiz_results qr
                  JOIN students st ON qr.student_id = st.id
                  LEFT JOIN classes c ON st.class = c.id
                  LEFT JOIN sections sec ON st.section = sec.id
                  WHERE qr.quiz_id = ?
                  ORDER BY qr.percentage DESC";
$stmt = mysqli_prepare($conn, $results_query);
mysqli_stmt_bind_param($stmt, "i", $quiz_id);
mysqli_stmt_execute($stmt);
$results_result = mysqli_stmt_get_result($stmt);

// Send Excel headers
header('Content-Type: application/vnd.ms-excel');
header('Content-Disposition: attachment; filename="test_results_' . date('Y-m-d') . '.xls"');

// Start table
echo '<table border="1">';
echo '<tr><th colspan="8" style="background-color: #4F81BD; color: white; font-size: 16px; height: 30px;">Test Results: ' . htmlspecialchars($selected_test['quiz_title']) . '</th></tr>';
echo '<tr><th colspan="8" style="background-color: #DCE6F1;">Subject: ' . htmlspecialchars($selected_test['subject_name']) . ' | Date: ' . date('M j, Y', strtotime($selected_test['start_date'])) . '</th></tr>';
echo '<tr style="background-color: #B8CCE4; font-weight: bold;">';
echo '<th>Student ID</th>';
echo '<th>Student Name</th>';
echo '<th>Class</th>';
echo '<th>Score</th>';
echo '<th>Percentage</th>';
echo '<th>Time Taken (seconds)</th>';
echo '<th>Status</th>';
echo '<th>Submitted At</th>';
echo '</tr>';

// Rows
while ($result = mysqli_fetch_assoc($results_result)) {
    echo '<tr>';
    echo '<td>' . htmlspecialchars($result['student_id']) . '</td>';
    echo '<td>' . htmlspecialchars($result['full_name']) . '</td>';
    echo '<td>' . htmlspecialchars($result['class_name'] . ' ' . $result['section_name']) . '</td>';
    echo '<td>' . $result['score'] . '/' . $result['total_marks'] . '</td>';
    echo '<td>' . $result['percentage'] . '%</td>';
    echo '<td>' . $result['time_taken'] . '</td>';
    echo '<td>' . ($result['result_released'] ? 'Released' : 'Pending') . '</td>';
    echo '<td>' . date('M j, Y g:i A', strtotime($result['submitted_at'])) . '</td>';
    echo '</tr>';
}

echo '</table>';
exit;
