<?php
include 'include/header.php';

// Enable output buffering
ob_start();

// Include database connection
include '../config/conn.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get user data from database
$user_id = $_SESSION['user_id'];
$query = "SELECT `id`, `name`, `email`, `password`, `role`, `school`, `image`, `created_at`, `password_updated_at` FROM users WHERE id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

// Initialize variables
$success = '';
$error = '';
$name = $user['name'];
$email = $user['email'];
$school = $user['school'];
$image = $user['image'];

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $school = trim($_POST['school'] ?? '');
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Validate required fields
    if (empty($name) || empty($email)) {
        $error = 'Name and email are required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Valid email is required';
    } else {
        // Check if email is already taken by another user
        $check_email = "SELECT id FROM users WHERE email = ? AND id != ?";
        $stmt_check = $conn->prepare($check_email);
        $stmt_check->bind_param("si", $email, $user_id);
        $stmt_check->execute();
        $result = $stmt_check->get_result();
        
        if ($result->num_rows > 0) {
            $error = 'Email is already taken by another account';
        } else {
            // Handle profile image upload
            $image_path = $user['image'];
            if (!empty($_FILES['profile_image']['name'])) {
                $target_dir = "../uploads/";
                if (!file_exists($target_dir)) {
                    mkdir($target_dir, 0777, true);
                }
                
                $file_extension = pathinfo($_FILES["profile_image"]["name"], PATHINFO_EXTENSION);
                $new_filename = "profile_" . $user_id . "_" . time() . "." . $file_extension;
                $target_file = $target_dir . $new_filename;
                
                // Check if image file is a actual image
                $check = getimagesize($_FILES["profile_image"]["tmp_name"]);
                if ($check !== false) {
                    if (move_uploaded_file($_FILES["profile_image"]["tmp_name"], $target_file)) {
                        // Delete old image if it exists
                        if (!empty($user['image']) && file_exists($user['image'])) {
                            unlink($user['image']);
                        }
                        $image_path = $target_file;
                    } else {
                        $error = 'Sorry, there was an error uploading your file.';
                    }
                } else {
                    $error = 'File is not an image.';
                }
            }
            
            // If there's no error, update the database
            if (empty($error)) {
                // Check if password is being updated
                if (!empty($current_password) && !empty($new_password)) {
                    if (!password_verify($current_password, $user['password'])) {
                        $error = 'Current password is incorrect';
                    } elseif ($new_password !== $confirm_password) {
                        $error = 'New passwords do not match';
                    } elseif (strlen($new_password) < 8) {
                        $error = 'New password must be at least 8 characters';
                    } else {
                        // Update with new password
                        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                        $update_query = "UPDATE users SET name = ?, email = ?, school = ?, image = ?, password = ?, password_updated_at = NOW() WHERE id = ?";
                        $stmt_update = $conn->prepare($update_query);
                        $stmt_update->bind_param("sssssi", $name, $email, $school, $image_path, $hashed_password, $user_id);
                        
                        if ($stmt_update->execute()) {
                            $success = 'Profile updated successfully';
                            // Update session variables
                            $_SESSION['user_name'] = $name;
                            $_SESSION['user_email'] = $email;
                            $stmt_update->close();
                        } else {
                            $error = 'Error updating profile: ' . $conn->error;
                        }
                    }
                } else {
                    // Update without changing password
                    $update_query = "UPDATE users SET name = ?, email = ?, school = ?, image = ? WHERE id = ?";
                    $stmt_update = $conn->prepare($update_query);
                    $stmt_update->bind_param("ssssi", $name, $email, $school, $image_path, $user_id);
                    
                    if ($stmt_update->execute()) {
                        $success = 'Profile updated successfully';
                        // Update session variables
                        $_SESSION['user_name'] = $name;
                        $_SESSION['user_email'] = $email;
                        $stmt_update->close();
                    } else {
                        $error = 'Error updating profile: ' . $conn->error;
                    }
                }
                
                // Refresh user data if update was successful
                if ($success) {
                    $query = "SELECT `id`, `name`, `email`, `password`, `role`, `school`, `image`, `created_at`, `password_updated_at` FROM users WHERE id = ?";
                    $stmt_refresh = $conn->prepare($query);
                    $stmt_refresh->bind_param("i", $user_id);
                    $stmt_refresh->execute();
                    $result = $stmt_refresh->get_result();
                    $user = $result->fetch_assoc();
                    $stmt_refresh->close();
                }
            }
        }
        $stmt_check->close();
    }
}

// Close database connection
$conn->close();
?>


    <style>
        .input-field:focus {
            box-shadow: 0 0 0 3px rgba(0, 150, 255, 0.2);
        }
        
        /* Toast notification styles */
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 16px 24px;
            border-radius: 8px;
            color: white;
            font-weight: 500;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            z-index: 1000;
            transform: translateX(150%);
            transition: transform 0.3s ease-in-out;
            display: flex;
            align-items: center;
            max-width: 350px;
        }
        .toast.show {
            transform: translateX(0);
        }
        .toast.success {
            background-color: #10B981;
        }
        .toast.error {
            background-color: #EF4444;
        }
        .toast.warning {
            background-color: #F59E0B;
        }
        .toast.info {
            background-color: #3B82F6;
        }
        .toast i {
            margin-right: 12px;
            font-size: 20px;
        }
        @media (max-width: 640px) {
            .toast {
                top: 10px;
                right: 10px;
                left: 10px;
                max-width: calc(100% - 20px);
            }
        }
        
        /* Profile image styles */
        .profile-image-container {
            position: relative;
            display: inline-block;
        }
        .profile-image {
            width: 120px;
            height: 120px;
            object-fit: cover;
            border: 4px solid white;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .profile-image-overlay {
            position: absolute;
            bottom: 0;
            right: 0;
            background: rgba(0, 150, 255, 0.8);
            color: white;
            border-radius: 50%;
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
        }
    </style>
</head>
<body class="font-sans bg-gray-50">

<!-- Toast Notification -->
<div id="toast" class="toast hidden">
    <i id="toastIcon"></i>
    <span id="toastMessage"></span>
</div>

<!-- Profile Page -->
<div class="min-h-screen p-4 py-8">
    <div class="max-w-4xl mx-auto">
        <div class="bg-white rounded-xl shadow-md p-6">
            <div class="flex justify-between items-center mb-6">
                <h1 class="text-2xl font-bold text-gray-800">Edit Profile</h1>
                <a href="profile.php" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-medium py-2 px-4 rounded-lg transition duration-200 flex items-center">
                    <i class="fas fa-arrow-left mr-2"></i> Back to Profile
                </a>
            </div>

            <?php if (!empty($error)): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6" role="alert">
                    <p class="font-bold">Error</p>
                    <p><?php echo htmlspecialchars($error); ?></p>
                </div>
            <?php endif; ?>

            <?php if (!empty($success)): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6" role="alert">
                    <p class="font-bold">Success</p>
                    <p><?php echo htmlspecialchars($success); ?></p>
                </div>
            <?php endif; ?>

            <form method="POST" action="" enctype="multipart/form-data" class="space-y-6">
                <!-- Profile Image -->
                <div class="flex flex-col items-center mb-6">
                    <div class="profile-image-container mb-4">
                        <img src="<?php echo !empty($user['image']) ? htmlspecialchars($user['image']) : 'https://ui-avatars.com/api/?name=' . urlencode($user['name']) . '&size=120&background=0096FF&color=fff'; ?>" 
                             alt="Profile Image" class="profile-image rounded-full">
                        <label for="profile_image" class="profile-image-overlay">
                            <i class="fas fa-camera"></i>
                        </label>
                    </div>
                    <input type="file" id="profile_image" name="profile_image" accept="image/*" class="hidden">
                    <p class="text-sm text-gray-500">Click on camera icon to change profile picture</p>
                </div>

                <!-- Personal Information -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label for="name" class="block text-sm font-medium text-gray-700 mb-1">Full Name</label>
                        <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($user['name']); ?>" 
                               class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary" required>
                    </div>

                    <div>
                        <label for="email" class="block text-sm font-medium text-gray-700 mb-1">Email Address</label>
                        <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" 
                               class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary" required>
                    </div>

                    <div>
                        <label for="school" class="block text-sm font-medium text-gray-700 mb-1">School</label>
                        <input type="text" id="school" name="school" value="<?php echo htmlspecialchars($user['school'] ?? ''); ?>" 
                               class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary">
                    </div>

                    <div>
                        <label for="role" class="block text-sm font-medium text-gray-700 mb-1">Role</label>
                        <input type="text" id="role" value="<?php echo htmlspecialchars(ucfirst($user['role'])); ?>" 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg bg-gray-100" disabled>
                    </div>
                </div>

                <!-- Password Change Section -->
                <div class="border-t border-gray-200 pt-6 mt-6">
                    <h2 class="text-lg font-medium text-gray-800 mb-4">Change Password</h2>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label for="current_password" class="block text-sm font-medium text-gray-700 mb-1">Current Password</label>
                            <input type="password" id="current_password" name="current_password" 
                                   class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary">
                        </div>

                        <div>
                            <label for="new_password" class="block text-sm font-medium text-gray-700 mb-1">New Password</label>
                            <input type="password" id="new_password" name="new_password" 
                                   class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary">
                        </div>

                        <div>
                            <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-1">Confirm New Password</label>
                            <input type="password" id="confirm_password" name="confirm_password" 
                                   class="input-field w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-primary">
                        </div>

                        <div class="flex items-end">
                            <p class="text-sm text-gray-500">Leave blank to keep current password</p>
                        </div>
                    </div>
                </div>

                <!-- Account Information -->
                <div class="border-t border-gray-200 pt-6 mt-6">
                    <h2 class="text-lg font-medium text-gray-800 mb-4">Account Information</h2>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Member Since</label>
                            <input type="text" value="<?php echo date('F j, Y', strtotime($user['created_at'])); ?>" 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg bg-gray-100" disabled>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Last Password Change</label>
                            <input type="text" value="<?php echo !empty($user['password_updated_at']) ? date('F j, Y', strtotime($user['password_updated_at'])) : 'Never changed'; ?>" 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg bg-gray-100" disabled>
                        </div>
                    </div>
                </div>

                <!-- Submit Button -->
                <div class="flex justify-end pt-6">
                    <button type="submit" class="bg-primary hover:bg-secondary text-white font-medium py-2 px-6 rounded-lg transition duration-200">
                        Update Profile
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
    // Toast notification function
    function showToast(message, type = 'success') {
        const toast = document.getElementById('toast');
        const toastIcon = document.getElementById('toastIcon');
        const toastMessage = document.getElementById('toastMessage');
        
        // Set toast content and style
        toastMessage.textContent = message;
        toast.className = 'toast'; // Reset classes
        toast.classList.add(type);
        
        // Set icon based on type
        switch(type) {
            case 'success':
                toastIcon.className = 'fas fa-check-circle';
                break;
            case 'error':
                toastIcon.className = 'fas fa-exclamation-circle';
                break;
            case 'warning':
                toastIcon.className = 'fas fa-exclamation-triangle';
                break;
            default:
                toastIcon.className = 'fas fa-info-circle';
        }
        
        // Show toast
        toast.classList.remove('hidden');
        toast.classList.add('show');
        
        // Hide after 5 seconds
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => toast.classList.add('hidden'), 300);
        }, 5000);
    }

    // Toggle password visibility
    function togglePassword(fieldId) {
        const field = document.getElementById(fieldId);
        const icon = field.nextElementSibling.querySelector('i');
        
        if (field.type === 'password') {
            field.type = 'text';
            icon.classList.remove('fa-eye');
            icon.classList.add('fa-eye-slash');
        } else {
            field.type = 'password';
            icon.classList.remove('fa-eye-slash');
            icon.classList.add('fa-eye');
        }
    }

    // Preview profile image before upload
    document.getElementById('profile_image').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                document.querySelector('.profile-image').src = e.target.result;
            }
            reader.readAsDataURL(file);
        }
    });

    // Show error/success messages from PHP
    <?php if (!empty($error)): ?>
        showToast('<?php echo addslashes($error); ?>', 'error');
    <?php endif; ?>

    <?php if (!empty($success)): ?>
        showToast('<?php echo addslashes($success); ?>', 'success');
    <?php endif; ?>
</script>

<?php
include 'include/footer.php';
ob_end_flush();
?>
